/**
 * Developed: 12/21/2015
 * Author: Aaron Melocik
 * 
 * BRIXSpinnerOverride is a book-level script to replace
 * the BRIX spinner (div.sham-spinner-blocker-brix) with
 * a notification element showing the user the BRIX's
 * type and guid.
 * 
 * This class is instantiated at the end of this file,
 * and that line should be removed prior to production.
 **/

/**
 * Any config items set to _false_
 * will not render at runtime.
 **/
var spinnerOverrideConfig = {
    parseBRIXTitle: true,
    parseBRIXChapter: true,
    parseBRIXAssignment: true,
    parseBRIXModuleNum: true,
    parseActivityURL: true,
    parseAssignmentURL: true,
    parseDataType: true,
    parseDataModule: true,
    parseDataContainerId: true,
    alwaysReplaceBRIXWithFallbackElement: true,
    secondsToCheckForBadSpinner: 15
};

var BRIXSpinnerOverride = function (configSettings) {
    this.config = configSettings || {};
    this.BRIXElementsSelector = document.querySelectorAll('div[data-containerid][data-assignmenturl]');
    this.BRIXElements = [];
    this.BRIXSpinner = document.querySelector('div.sham-spinner-blocker-brix');
    this.notificationElementStyleString = 'display: block; width: 80%; margin: 1em 1em 1em 1em; color: #2500BF; padding: .25em .25em .25em 1.25em; border: double 1px red;';

    // Create an array of BRIX elements and properties, then update elements on the page.
    this.getBRIXProperties();

    // If we only want to run this code in the presence of a spinner element...
    if (!this.config.alwaysReplaceBRIXWithFallbackElement) {
        this.overrideSpinnerAndBRIX();
    }
}; // end BRIXSpinnerOverride()


/**
 * Grab element references and properties from the
 * BRIX elements. Push element + property objects
 * on BRIXElements array.
 */
BRIXSpinnerOverride.prototype.getBRIXProperties = function () {
    for (var i = 0; i < this.BRIXElementsSelector.length; i++) {
        // Create object to store element data
        var thisBRIXElement = {};

        // Get element and property values
        thisBRIXElement.element = this.BRIXElementsSelector[i];
        thisBRIXElement.parentElement = thisBRIXElement.element.parentElement;
        thisBRIXElement.brixTitle = thisBRIXElement.element.getAttribute('data-brix-title-for-fallback');
        thisBRIXElement.brixChapter = thisBRIXElement.element.getAttribute('data-brix-chapter-for-fallback');
        thisBRIXElement.brixAssignment = thisBRIXElement.element.getAttribute('data-brix-assignment-for-fallback');
        thisBRIXElement.brixModuleNum = thisBRIXElement.element.getAttribute('data-brix-modulenum-for-fallback');
        thisBRIXElement.activityURL = thisBRIXElement.element.getAttribute('data-activityurl');
        thisBRIXElement.assignmentURL = thisBRIXElement.element.getAttribute('data-assignmenturl');
        thisBRIXElement.dataModule = thisBRIXElement.element.getAttribute('data-module');
        thisBRIXElement.dataContainerId = thisBRIXElement.element.getAttribute('data-containerid');
        thisBRIXElement.dataType = thisBRIXElement.element.getAttribute('data-type');

        // Split and hash activity and assignment URLs
        if (!!thisBRIXElement.activityURL) {
            var arrayOfActivityURLDirs = thisBRIXElement.activityURL.split('/');
            thisBRIXElement.activityHash = arrayOfActivityURLDirs[(arrayOfActivityURLDirs.length - 1)];
        }

        if (!!thisBRIXElement.assignmentURL) {
            var arrayOfAssignmentURLDirs = thisBRIXElement.assignmentURL.split('/');
            thisBRIXElement.assignmentHash = arrayOfAssignmentURLDirs[(arrayOfAssignmentURLDirs.length - 1)];
        }

        // Push this element onto the array
        this.BRIXElements.push(thisBRIXElement);

        // Add title property to BRIX elements to see guid on hover
        this.addTitle(this.BRIXElements[i]);

        // Create notification element (placeholder with guid info)
        this.createNotificationElement(this.BRIXElements[i]);
    } // end for (BRIX elements)
}; // end getBRIXProperties()


/**
 * Add a 'title' property to the top-level BRIX element
 * tag. This way Book Dev can mouse over the element
 * and see the guid and type even if the element renders.
 */
BRIXSpinnerOverride.prototype.addTitle = function (thisBRIXElement) {
    thisBRIXElement.titleString = '';

    if (!!thisBRIXElement.brixTitle) {
        thisBRIXElement.titleString += 'BRIX Title: ' + thisBRIXElement.brixTitle;
    }

    if (!!thisBRIXElement.activityHash) {
        thisBRIXElement.titleString += '\
        activity: ' + thisBRIXElement.activityHash;
    }

    if (!!thisBRIXElement.assignmentHash) {
        thisBRIXElement.titleString += '\
        assignment: ' + thisBRIXElement.assignmentHash;
    }

    // Set the element's title for a tooltip effect.
    thisBRIXElement.element.setAttribute('title', thisBRIXElement.titleString);
}; // end addTitle()


/**
 * Create DOM element with BRIX element's guid and type.
 */
BRIXSpinnerOverride.prototype.createNotificationElement = function (thisBRIXElement) {
    // Create notification element container
    thisBRIXElement.notificationElement = document.createElement('div');
    thisBRIXElement.notificationElement.setAttribute('style', this.notificationElementStyleString);

    // Create and append child elements
    if (thisBRIXElement.brixTitle && this.config.parseBRIXTitle) {
        var brixTitleTextElement = document.createElement('p');
        brixTitleTextElement.textContent = 'BRIX Title (A.K.A. Module): ' + thisBRIXElement.brixTitle;
        thisBRIXElement.notificationElement.appendChild(brixTitleTextElement);
    }
    if (thisBRIXElement.brixChapter && this.config.parseBRIXChapter) {
        var brixChapterTextElement = document.createElement('p');
        brixChapterTextElement.textContent = 'BRIX Chapter: ' + thisBRIXElement.brixChapter;
        thisBRIXElement.notificationElement.appendChild(brixChapterTextElement);
    }
    if (thisBRIXElement.brixAssignment && this.config.parseBRIXAssignment) {
        var brixAssignmentTextElement = document.createElement('p');
        brixAssignmentTextElement.textContent = 'BRIX Assignment: ' + thisBRIXElement.brixAssignment;
        thisBRIXElement.notificationElement.appendChild(brixAssignmentTextElement);
    }
    if (thisBRIXElement.brixModuleNum && this.config.parseBRIXModuleNum) {
        var brixModuleNumTextElement = document.createElement('p');
        brixModuleNumTextElement.textContent = 'BRIX Module Number: ' + thisBRIXElement.brixModuleNum;
        thisBRIXElement.notificationElement.appendChild(brixModuleNumTextElement);
    }
    if (thisBRIXElement.activityHash && this.config.parseActivityURL) {
        var activityTextElement = document.createElement('p');
        activityTextElement.textContent = 'Activity ID: ' + thisBRIXElement.activityHash;
        thisBRIXElement.notificationElement.appendChild(activityTextElement);
    }
    if (thisBRIXElement.assignmentHash && this.config.parseAssignmentURL) {
        var assignmentTextElement = document.createElement('p');
        assignmentTextElement.textContent = 'Assignment ID: ' + thisBRIXElement.assignmentHash;
        thisBRIXElement.notificationElement.appendChild(assignmentTextElement);
    }
    //if (thisBRIXElement.dataModule && this.config.parseDataModule) {
        //var dataModuleTextElement = document.createElement('p');
        //dataModuleTextElement.textContent = 'Data Module: ' + thisBRIXElement.dataModule;
        //thisBRIXElement.notificationElement.appendChild(dataModuleTextElement);
    //}
    if (thisBRIXElement.dataContainerId && this.config.parseDataContainerId) {
        var containerIdTextElement = document.createElement('p');
        containerIdTextElement.textContent = 'Container ID: ' + thisBRIXElement.dataContainerId;
        thisBRIXElement.notificationElement.appendChild(containerIdTextElement);
    }
    // if (thisBRIXElement.dataType && this.config.parseDataType) {
    //     var dataTypeTextElement = document.createElement('p');
    //     dataTypeTextElement.textContent = 'Item Type: ' + thisBRIXElement.dataType;
    //     thisBRIXElement.notificationElement.appendChild(dataTypeTextElement);
    // }

    thisBRIXElement.parentElement.appendChild(thisBRIXElement.notificationElement);
}; // end createNotificationElement()


/**
 * Hide the spinner and BRIX elements and add an
 * event listener to restore the BRIX if it loads
 * correctly by watching for the spinner's 'display'.
 */
BRIXSpinnerOverride.prototype.overrideSpinnerAndBRIX = function () {
    // If there is no BRIX spinner on the page
    if (!this.BRIXSpinner) {
        console.warn('WARNING: there is no BRIX Spinner element on the page!');
    } else {
        // If spinner is not already hidden, override.
        debugger;
        if (!this.BRIXElementHasLoaded()) {
            // Get current spinner style and convert 'null' string to ''
            var spinnerStyle = this.BRIXSpinner.getAttribute('style');
            if (!spinnerStyle) {
                spinnerStyle = '';
            }

            // Hide the spinner
            this.BRIXSpinner.setAttribute('style', (spinnerStyle + ' visibility: hidden;'));

            // Hide the BRIX elements
            for (var j = 0; j < this.BRIXElements.length; j++) {
                this.BRIXElements[j].element.style.visibility = 'hidden';
            }

            // Begin the recursive function to continue checking for BRIX status
            this.updateNotificationElements();
        } // end if (spinner is not hidden)
    } // end else (spinner element exists on page)
}; // end overrideSpinnerAndBRIX()


/**
 * Check spinner's style attribute for "display: hidden",
 * which indicates that the BRIX element has loaded successfully.
 */
BRIXSpinnerOverride.prototype.BRIXElementHasLoaded = function () {
    // If we have manually set this.BRIXSpinner as a flag, return false to continue rendering.
    if (this.BRIXSpinner === true) {
        return false;
    } else {
        this.BRIXSpinner.spinnerStyle = this.BRIXSpinner.getAttribute('style');

        // If spinner is already hidden, return 'true'.
        if (!this.BRIXSpinner.spinnerStyle || this.BRIXSpinner.spinnerStyle.search('display: none') === -1) {
            return false;
        } else {
            return true;
        }
    } // end else (this.BRIXSpinner is not being used as a flag)
}; // end BRIXElementHasLoaded()


/**
 * Check to see if the BRIX element has properly
 * loaded. If so, remove the notification element
 * and display the BRIX element.
 */
BRIXSpinnerOverride.prototype.updateNotificationElements = function () {
    this.config.secondsToCheckForBadSpinner--;

    // Update notifications and BRIX elements
    if (this.BRIXElementHasLoaded()) {
        // Update all BRIX elements on page
        for (var i = 0; i < this.BRIXElements.length; i++) {
            // Remove notification element
            this.BRIXElements[i].notificationElement.remove();

            // Show BRIX element
            this.BRIXElements[i].element.style.visibility = 'visible';
        }
    } else {
        // If we haven't already done our max number of checks...
        if (this.config.secondsToCheckForBadSpinner > 0) {
            window.setTimeout(this.updateNotificationElements.bind(this), 1000);
        } // end if (this.config.secondsToCheckForBadSpinner >= 0)
    }
}; // end updateNotificationElements()

window.onload = function () {
    var spinnerOverride = new BRIXSpinnerOverride(spinnerOverrideConfig);
}; // end window.onload