/**
 * Author: Aaron Melocik
 * Major Revision: 3.31.2015
 *
 * md.userDevice Properties:
 *   devicePixelRatio
 *   type (pc, mobile, tablet)
 *   isMobile
 *   isTablet
 *   isPC
 *   orientation
 *   windowWidth
 *   windowHeight
 *   userAgentString
 *   fullscreen
 *   hasTouch
 *
 * md.userDevice Methods:
 *   getDocumentBodyWidth
 *   polyfillDeviceObjects
 *   getUserDevice
 *   getOrientation
 *   fullscreenEnabled
 *   requestFullscreen
 *   exitFullscreen
 *   md.userDevice.lockOrientation()
 *
 * userDevice object is added to the main 'md' object, and will:
 *   1) Expose orientation API through polyfilling for old browsers
 *   2) Expose fullscreen API through polyfilling for old browsers
 *   3) Expose properties about the user's screen and device
 *
 * REFERENCES:
 * 1) Screen Orientation Specification: http://www.w3.org/TR/screen-orientation/
 * 2) Device Orientation Specification: http://w3c.github.io/deviceorientation/spec-source-orientation.html
 * 3) Fullscreen library: https://github.com/sindresorhus/screenfull.js/
 * 4) Lock orientation (see example @ bottom): http://www.sitepoint.com/introducing-screen-orientation-api/
 *                                             https://developer.mozilla.org/en-US/docs/Web/API/CSS_Object_Model/Managing_screen_orientation
 *
 * FUTURE CONSIDERATIONS:
 * 1) Inject <meta name='viewport' content='width=device-width, height=device-height, initial-scale=1, minimum-scale=0, maximum-scale=2, user-scalable=yes, target-densitydpi=device-dpi' /> into head if mobile is detected???
 * 2) Consider other DETECTION OPTIONS:
 *     A. Script(s) below
 *     B. Apache Mobile filter
 *     C. wurfl
 *     D. detectright
 **/


//var md = md || {};
//md.userDevice = new Class({
var UserDevice = new Class({
    Implements: [Options],
    options: {
        debuggingMode: false // set as 'true' for debugging messages
    },
    initialize: function (options) {
        this.setOptions(options);
        var _this = this;

        /**
         * Useful variables
         **/
        _this.devicePixelRatio = window.devicePixelRatio;
        _this.type = null; // can be 'tablet', 'mobile', 'pc'
        _this.isMobile = false;
        _this.isTablet = false;
        _this.isPC = false;

        _this.orientation = undefined;
        _this.windowWidth = window.getWidth();
        // _this.windowWidth = $(window).width();
        _this.windowHeight = window.getHeight();
        // _this.windowHeight = $(window).height();

        _this.userAgentString = navigator.userAgent.toLowerCase();
        _this.fullscreen = undefined;
        _this.fullscreenEnabled = null;
        _this.requestFullscreen = null;
        _this.exitFullscreen = null;
        _this.hasTouch = null;

        /**
         * Possibly-useful variables, kept during development, but can be removed closer to release.
         **/
        _this.screenWidth = screen.width;
        _this.screenHeight = screen.height;
        _this.pixelDepth = window.pixelDepth; // Firefox only
        _this.availableScreenWidth = window.screen.availWidth; // this is the total screen real estate, and is larger than getWidth()
        _this.availableScreenHeight = window.screen.availHeight; // this is the total screen real estate, and is larger than getHeight()
        _this.documentBodyWidth = document.body.getStyle('width');
        _this.getDocumentBodyWidth = function () {
            _this.documentBodyWidth = document.body.getStyle('width');
        };

        _this.polyfillDeviceObjects();
        _this.getUserDevice();
    }, // end initialize()


    /**
     * Detect screen/device orientation API.
     * If not up to W3C proposed standards, polyfill the parts we need.
     *
     * For orientation, test progressively for:
     *  - screen.orientation / screen.mozOrientation / screen.msOrientation;
     *  - mediaQuery
     *  - W & H comparison
     *
     * Exposes:
     *   md.userDevice.getOrientation()
     *   md.userDevice.orientation.angle,
     *   md.userDevice.orientation.onchange,
     *   md.userDevice.orientation.type
     *     (can be: landscape-primary, landscape-secondary, portrait-secondary, portrait-primary)
     *   md.userDevice.requestFullscreen()
     *   md.userDevice.exitFullscreen()
     *   md.userDevice.fullscreenEnabled()
     *   md.userDevice.lockOrientation()
     **/
    polyfillDeviceObjects: function () {
        var _this = this;

        /**
         * Assign md.userDevice.orientation and add listeners if necessary.
         * For orientation objects already attached to screen, should expose method _this.orientation.onchange,
         * but we handle this below for added safety.
         **/
        if (window.screen.orientation) {
            _this.orientation = window.screen.orientation;
            _this.getOrientation = function () {
                _this.orientation = window.screen.orientation;
                return _this.orientation.type;
            }
        } else if (window.screen.mozOrientation) {
            _this.orientation = window.screen.mozOrientation;
            _this.getOrientation = function () {
                _this.orientation = window.screen.mozOrientation;
                return _this.orientation.type;
            }
        } else if (window.screen.msOrientation) {
            _this.orientation = window.screen.msOrientation;
            _this.getOrientation = function () {
                _this.orientation = window.screen.msOrientation;
                return _this.orientation.type;
            }
        } else {
            console.warn('%c No support for screen orientation API in this browser. Creating custom screen object...', 'color: #d10000');

            _this.orientation = {
                angle: 0,
                onchange: null, // TODO: add events based on listening to screen W & H, and/or MediaQueries.
                type: null
            };
            _this.getOrientation = function () {
                this.orientation.type = (function () {
                        /**
                         * Fallback to meqia query list: https://developer.mozilla.org/en-US/docs/Web/Guide/CSS/Testing_media_queries
                         **/
                        if (window.matchMedia('(orientation: portrait)')) {
                            return 'portrait';
                        } else if (window.matchMedia('(orientation: landscape)')) {
                            return 'landscape';
                        } else { // end media query list checking
                            /**
                             * The device orientation is not explicitly defined. Falling back to detecting W & H.
                             **/
                            console.warn('device orientation is not defined from media query. Comparing Width & Height...');
                            windowWidth

                            if (_this.windowWidth > _this.windowHeight) {
                                if (_this.debuggingMode) {
                                    console.log('screen is wider than tall; orientation is in LANDSCAPE');
                                }
                                return 'landscape';
                            } else if (_this.windowWidth < _this.windowHeight) {
                                if (_this.debuggingMode) {
                                    console.log('screen is taller than wide; orientation is in PORTRAIT');
                                }
                                return 'portrait';
                            } else {
                                console.error('cannot determine orientation of screen. Possibly it\'s a square?');
                                return false;
                            }
                        } // end (compare screen width to height)
                    })() // end immediate invocation of anonymous orientation-evaluating function
                return _this.orientation.type;
            }; // end _this.getOrientation() function definition
        } // end else (no window.screen)

        _this.orientation.previousType = _this.orientation.type;

        if (_this.debuggingMode) {
            console.log(_this.orientation);
        }

        /**
         * Provide interface for browser-neutral requestFullscreen(element) and exitFullscreen methods
         **/
        if (document.body.requestFullscreen) {
            _this.requestFullscreen = function (element) {
                element.requestFullscreen();
                return element;
            }; // end _this.requestFullScreen(element)
        } else if (document.body.webkitRequestFullscreen) {
            _this.requestFullscreen = function (element) {
                element.webkitRequestFullscreen();
                return element;
            }; // end _this.requestFullScreen(element)
        } else if (document.body.mozRequestFullScreen) {
            _this.requestFullscreen = function (element) {
                element.mozRequestFullScreen();
                return element;
            }; // end _this.requestFullScreen(element)
        } else if (document.body.msRequestFullscreen) {
            _this.requestFullscreen = function (element) {
                element.msRequestFullscreen();
                return element;
            }; // end _this.requestFullScreen(element)
        } else {
            console.warn('WARNING: no requestFullscreen detected...');
        } // end requestFullscreen method assignment

        if (document.exitFullscreen) {
            _this.exitFullscreen = document.exitFullscreen;
        } else if (document.mozCancelFullScreen) {
            _this.exitFullscreen = document.mozCancelFullScreen;
        } else if (document.webkitExitFullscreen) {
            _this.exitFullscreen = document.webkitExitFullscreen;
        } else if (document.msExitFullscreen) {
            _this.exitFullscreen = document.msExitFullscreen;
        } else {
            console.warn('WARNING: no exitFullscreen detected...');
        } // end exitFullscreen method assignment

        if (document.fullscreenEnabled) {
            _this.fullscreenEnabled = document.fullscreenEnabled;
        } else if (document.mozFullScreenEnabled) {
            _this.fullscreenEnabled = document.mozFullScreenEnabled;
        } else if (document.webkitFullscreenEnabled) {
            _this.fullscreenEnabled = document.webkitFullscreenEnabled;
        } else if (document.msFullscreenEnabled) {
            _this.fullscreenEnabled = document.msFullscreenEnabled;
        } else {
            console.warn('WARNING: no exitFullscreen detected...');
        } // end fullscreenEnabled method assignment


        /**
         * lockOrientation accepts one of the following strings:
         *   'natural' (or possibly 'default')
         *   'portrait'
         *   'landscape'
         *   'portrait-primary'
         *   'portrait-secondary'
         *   'landscape-primary'
         *   'landscape-secondary'
         *
         * Invoke with _this.lockOrientation('landscape').
         * Note that the device may have to be in 'fullscreen' for this to work.
         **/
        if (screen.lockOrientation) {
            _this.lockOrientation = screen.lockOrientation;
        } else if (screen.mozLockOrientation) {
            _this.lockOrientation = screen.mozLockOrientation;
        } else if (screen.msLockOrientation) {
            _this.lockOrientation = screen.msLockOrientation;
        } else {
            console.warn('WARNING: no lockOrientation possible.');
        } // end fullscreenEnabled method assignment                
    }, // end polyfillDeviceObjects()

    /**
     * Register events for orientation changes
     * Orientation info at https://developer.mozilla.org/en-US/docs/Web/API/Detecting_device_orientation
     *                 and https://developer.mozilla.org/en-US/docs/Web/Guide/CSS/Testing_media_queries
     *
     * Handle with either addEventListener('deviceorientation') or window.matchMedia('(orientation: [[portrait) || landscape]]')
     * Invoke with _this.addOrientationChangeEvent(function);
     **/
    addOrientationChangeEvent: function (orientationEventFunction) {
        try {
            screen.addEventListener("orientationchange", orientationEventFunction);
        } catch (error) {
            try {
                window.addEventListener("deviceorientation", function (orientationEventFunction) {
                    _this.orientation.previousType = _this.orientation.type;
                    md.userDevice.getOrientation();
                    if (_this.orientation.type !== _this.orientation.previousType) {
                        orientationEventFunction();
                    }
                }, true); // end addEventListener(window deviceorientation)
            } catch (error) {
                // Brute force after resize requiring JQuery
                $(window).resize(orientationEventFunction);
            } // end 2nd try-catch
        } // end 1st try-catch
    }, // end addOrientationChangeEvent()

    removeOrientationChangeEvent: function (orientationEventFunction) {
        try {
            screen.removeEventListener("orientationchange", orientationEventFunction);
        } catch (error) {
            try {
                window.removeEventListener("deviceorientation", function (orientationEventFunction) {
                    _this.orientation.previousType = _this.orientation.type;
                    md.userDevice.getOrientation();
                    if (_this.orientation.type !== _this.orientation.previousType) {
                        orientationEventFunction();
                    }
                }, true); // end addEventListener(window deviceorientation)
            } catch (error) {
                // Nothing to see here...
            } // end 2nd try-catch
        } // end 1st try-catch
    }, // end removeOrientationChangeEvent()


    /**
     * Get physical and API information about the user's device.
     **/
    getUserDevice: function () {
        //        this.listenForOrientationEvents();
        this.parseUserAgentString();
        this.detectTouchCapability();
    }, // end getUserDevice()

    /**
     * Sniff user agent string searching for device indication
     **/
    parseUserAgentString: function () {
        if (this.debuggingMode) {
            console.log('%c navigator.userAgent =' + navigator.userAgent, 'color: green');
        }

        if (/mobi|android|touch|mini/i.test(this.userAgentString)) {
            if (this.debuggingMode) {
                console.log('detected string "mobi" in userAgent');
            }
            this.isMobile = true;
            this.type = 'mobile';
        } else if (/tabl/i.test(this.userAgentString)) {
            if (this.debuggingMode) {
                console.log('detected string "tabl" in userAgent');
            }
            this.isTablet = true;
            this.type = 'tablet';
        } else {
            console.warn('%c No "mobi" or "tabl" detected in userAgent string. Current device considered a PC.', 'color: purple');
            this.isPC = true;
            this.type = 'pc';
        }
    }, // end parseUserAgentString()


    detectTouchCapability: function () {
            var _this = this;
            // TODO
            _this.hasTouch = false;

            if ('ontouchstart' in window || window.navigator.msMaxTouchPoints in window) {
                _this.hasTouch = true;
            }; // works on ie10

            if (_this.hasTouch && this.isPC) {
                if (_this.debuggingMode) {
                    console.log('ambiguous device details - seems to be a PC, but also has touch events...');
                }
            }
        } // end detectTouchCapability()
    
    

    //    deviceRotationListener: function(timeoutDelayValue) {
    //        var _this = this;
    //        
    //        console.log('in setTimeoutFunction');
    //
    //        if (_this.mask.element.getHeight() > _this.mask.element.getWidth()) {
    //            _this.viewportDimensions.orientation = "portrait";
    //        } else {
    //            _this.viewportDimensions.orientation = "landscape";
    //        }
    //
    //        // orientation has changed, so toggle the warning notification & widget
    //        if (_this.viewportDimensions.orientation === _this.modal.orientation) {
    //            // orientations are the same, so if the notification is already visible, hide it.
    //            if (_this.orientationNotificationElement.getStyle('display') !== 'none') {
    //*****    //                _this.modal.modalBody.iframeEl.show();
    //                _this.orientationNotificationElement.hide();
    //            }
    //        } else if (_this.orientationNotificationElement.getStyle('display') === 'none'){
    //            // orientations are the different, so if the notification is not already visible, display it.
    //*****    //            _this.modal.modalBody.iframeEl.hide();
    //            _this.orientationNotificationElement.show();
    //        }
    ////        debugger;
    //        setTimeout(_this.deviceRotationListener(timeoutDelayValue), timeoutDelayValue, _this);
    //    }, // end deviceRotationListener()
    
    
}); // end UserDevice


var md = md || {};
md.userDevice = new UserDevice()