/*
---

script: SideBar.js

description: Provides the TOC UI.

requires:
 - Touch
 - Core/Fx
 - More/Assets
 - Behavior/Behavior
 - More/Object.Extras

provides: [TOC, TOC.Nav]

...
*/

var ThreeSixty = new Class({
	Implements: [Options, Events],
	options: {
		/*
			onProgress: function(percent, index, source){
				console.log('loaded ' + percent + '% index: ' + index);
			},
			ready: function(){
				//all images loaded
			},
			//if no image is defined as already in the DOM, one will be injected in the container
			image: elmentIdOrReference,
		*/
		images: {
			//how many images
			count: 72,
			//path to images with ## for number values; note that this is padded with a zero, so 01-09, 10, 11, etc
			path: 'images/Seq_v04_640x378_##.jpg'
		},
		//when the user flicks, how many rotations can the thing go (maximum) before it stops
		maxFlickMultiplier: 1,
		//how far the flick has to be for there to be a transition at all on flick
		minFlick: 5,
		//the distance in pixels it takes to rotate the view; controls how "fast" it feels
		rotateDistance: 1000,
		//play the animation on startup?
		playIntro: true,
		//Determine whether image is full 360 or not
		full360: true
	},
	/*
		state variables
	*/
	//the current frame
	current: 0,
	//a history of positions, updated periodically; used to compute the "flick" velocity
	history: [0],
	//the position when a drag event starts
	dragStartPos: 0,
	//array of image sources for our animation (computed by computeSources method)
	sources: [],

	initialize: function(container, options){
		this.element = document.id(container);
		this.setOptions(options);
		this.attach();
		this.computeSources();
		this.makeImg();
		this.makeFx();
		this.preload();
	},
	//get a reference to or create an image for our animation
	makeImg: function(){
		if (this.options.image) this.img = document.id(this.options.image);
		if (!this.img){
			//create a new image and inject it in our target if there isn't one
			this.img = new Element('img', {
				src: this.sources[0]
			}).inject(this.element);
		}
	},
	makeFx: function(){
		//create an fx object for spinning
		this.fx = new Fx({
			transition: Fx.Transitions.Quad.easeOut,
			fps: 120
		});
		//define the set method for our Fx instance as rotating the image
		this.fx.set = function(i){
			this.show(i.round());
		}.bind(this);
	},
	//define the sources for all the images in the set; put them in this.sources array
	computeSources: function(){
		var path = this.options.images.path.split('##');
		this.options.images.count.times(function(i){
			this.sources.push(path[0] + this.pad(i + 1, 2) + path[1]);
		}, this);
		return this.sources;
	},
	//preload all the images before we run the animation
	preload: function(){
		Asset.images(this.sources, {
			onProgress: function(counter, index, source){
				this.fireEvent('progress', [((counter / this.options.images.count) * 100).round(), index, source])
			}.bind(this),
			onComplete: function(){
				this.fireEvent('ready');
				if (this.options.playIntro) this.intro();
			}.bind(this)
		})
	},
	//play the intro animation
	intro: function(){
		//play a little slower for the intro
		this.fx.options.duration = 1000;
		//start the animation
		this.fx.start(this.current, this.options.images.count).chain(function(){
			//when it finishes, put the duration back
			this.fx.options.duration = 500;
			//set the current frame
			this.current = this.options.images.count;
		}.bind(this));
	},
	//method to pad a number a given digits
	//i.e. this.pad(1, 2, '0') returns '01'
	pad: function(n, digits, string){
		if (digits == 1) return n;
		return n < Math.pow(10, digits - 1) ? (string || '0') + this.pad(n, digits - 1, string) : n;
	},
	//attach the drag event logic; uses Touch.js for this
	attach: function(){
		this.touch = new Touch(this.element).addEvents({
			start: this.startDrag.bind(this),
			end: this.endDrag.bind(this),
			move: this.drag.bind(this)
		});
	},
	//fired when dragging starts
	startDrag: function(){
		//cancel any running effect (a flick or the intro)
		this.fx.cancel();
		//set the start position
		this.dragStartPos = this.current;
		//reset the history
		this.history.empty();
		this.dragOffset = 0;
		//push the location into the history every 15 ms; used to compute flick
		this.dragInterval = this.updateDragHistory.periodical(10, this);
		this.updateDragHistory();
	},
	//updates the drag position in the history
	updateDragHistory: function(){
		//put the current position at the beginning
		this.history.unshift(this.dragOffset);
		//keep the array at 3 long for teh memories
		if (this.history.length > 3) this.history.pop();
	},
	//on drag end, clear the interval and fire the flick
	endDrag: function(){
		clearInterval(this.dragInterval);
		//get the distance moved from the history (30 ms ago)
		var flick = (this.history.getLast() - this.dragOffset);
		//get the size of the container
		var size = this.element.getSize().x;
		//rotate based on the relative size of the container,
		//i.e. if you moved 1x the size of the container in 30ms
		//rotate 1 full time. if you moved only .5x, rotate a half turn, etc.
		var rotation = (flick / size) * this.options.images.count;
		this.flick(rotation.round());
	},
	//drag handler; figures out rotation position based on drag offset
	//x = integer of drag offset
	drag: function(x){
		this.dragOffset = x;
		//compute the frame distance as the drag distance devided by the distance one must drag for a full rotation
		var percentDiffX = x / this.options.rotateDistance,
				//and the frame we're on is the image count times that percent offset
				frameDiffX = (this.options.images.count * percentDiffX).round();
		//now rotate the negative of that number, because in iOS we follow the finger, not the "scroll bar" as on the web
		this.rotate(-frameDiffX);
	},
	//rotate the image by x frames
	rotate: function(offset){
		if(this.options.full360) {
			//compute the image index to display
			//get the offset from the start position and wrap it at the image count size
			var index = (offset + this.dragStartPos) % this.options.images.count;
			if (index < 0) index = this.options.images.count + index;
			this.show(index);
		} else {
			//Do not allow rotation
			var index = offset + this.dragStartPos;
			if(index >= 0 && index < this.options.images.count) {
				this.show(index);
			}
		}
		
	},
	show: function(index){
		//handle the possibility that something has gone wrong and we're trying to set an image source that isn't valid
		if (this.sources[index]) {
			//set the image source
			this.img.src = this.sources[index];
			//update the current position
			this.current = index;
			this.fireEvent('changed', {'index': this.current});
		}
	},
	//animate the rotation a given distance (number of frames) by a factor of 2
	flick: function(distance){
		//if it's not at least the minFlick value, exit
		if (distance.abs() < this.options.minFlick) return;
		//if it's greater than the maxFlickMultiplier * the image count, use that max value
		if (distance.abs() > this.options.maxFlickMultiplier * this.options.images.count) {
			distance = this.options.maxFlickMultiplier * this.options.images.count * (distance > 0 ? 1 : -1);
		}
		//if the distance value was postive, make the dist value negative to get the spin direction correct
			this.fx.start(this.current, this.current + distance);
	}
});

//some cross-lib compatibility; Touch.js and MooTools 1.3 aren't quite in sync
if (Browser.Platform.ios) Browser.Platform.ipod = true;


Behavior.addGlobalFilters({
	ThreeSixty: {
		defaults: {
			playIntro: false,
			image: 'img.frame'
		},
		setup: function(element, api){
			var ts = new ThreeSixty(element,
				Object.merge(
					{
						image: element.getElement(api.get('image'))
					},
					Object.cleanValues(
						api.getAs({
							images: Object,
							playIntro: Boolean
						})
					)
				)
			);
			return ts;
		}
	}
});