/*
---

script: metrodigi-drag.js

description: Provides Drag-drop based Games structure

requires: [mootools.touch, mootools.drag, More/Array.Extras]

provides: [md.dragdrop]

...
*/

var md = md || {};
md.dragdrop = md.dragdrop || {};
/* Simple Class to display and calculate results.*/
md.dragdrop.ResultManager = new Class({
    Implements: [Options, Events],
    //DEFAULT OPTIONS
    options: {
        container: undefined,
        show: false,
        correctCountSelector: '.correct span',
        incorrectCountSelector: '.incorrect span'
    },
    initialize: function (questionCount, options) {
        if (typeOf(questionCount) != 'number') {
            throw("questionCount must be number. received: " + questionCount);
        }
        this.questionCount = questionCount;
        this._correctCount = 0;
        this._incorrectCount = 0;
        this.setOptions(options);
    },
    setView: function (selector, value) {
        if (this.options.show && this.options.container != undefined) {
            this.options.container.getElement(selector).set('text', value);
        }
    },
    incrementCorrect: function () {
        this._correctCount++;
        this.setView(this.options.correctCountSelector, this._correctCount);
        if (this._correctCount == this.questionCount) {
            this.fireEvent('allQuestionsCorrect', {correctPercentage: this.getCorrectPercentage()});
        }
    },
    incrementIncorrect: function () {
        this._incorrectCount++;
        this.setView(this.options.incorrectCountSelector, this._incorrectCount);
    },
    getCorrectPercentage: function () {
        var totalTrial = this._correctCount + this._incorrectCount;
        return this.questionCount / totalTrial * 100;
    }
});
md.dragdrop.BaseGame = new Class({
    Implements: [Options, Events],
    //DEFAULT OPTIONS
    options: {
        correctClass: 'correct',
        incorrectClass: 'incorrect',
        hoverClass: 'dragHover',
        finishMsgSelector: ".finishMsg",
        errorAudioPath: '',
        // errorAudioPath:'../Audio/error.m4a',
        isCorrectCheck: undefined,
        result: {}
    },
    initialize: function (containerEl, options) {
        this.reflowable = document.body.hasClass('book-type-reflowable') && !(window.parent && window.parent.Lindgren);
        if (this.reflowable) {
            this.refactorDrag();
        }
        if (typeOf(options) == 'null' ||
                typeOf(options.draggables) == 'null' || typeOf(options.droppables) == 'null' || typeOf(options.draggablesContainer) == 'null') {
            throw('options.draggables & options.droppables & options.draggablesContainer are compulatory options.');
        }
        this.setOptions(options);
        if (this.options.errorAudioPath !== undefined) {
            this.errorAudio = new Audio(this.options.errorAudioPath);
        }
        this.setupDraggables();
    },
    refactorDrag: function () {
        // Make drag events use client instead of page coordinates.
        Class.refactor(Drag, {
            start: function (e) {
                e.page = e.client;
                this.previous(e);
            },
            drag: function (e) {
                e.page = e.client;
                this.previous(e);
            }
        });
    },
    startDragHandler: function (context) {
        return function (event) {
            var self = context;
            event.stop();
            var draggable = this;
            var coordinates = {
                left: draggable.getLeft(),
                top: draggable.getTop(),
                width: draggable.getComputedStyle('width'),
                height: draggable.getComputedStyle('height')
            };
            var clone = draggable.clone().setStyles(coordinates).setStyles({
                opacity: 0.7,
                position: 'absolute',
                'z-index': 11000
            }).inject(self.options.draggablesContainer);
            clone.orig = draggable;
            var container = self.options.appContainer || $$('body'),
                    dragOptions = {
                        container: container,
                        droppables: self.options.droppables,
                        onDrop: self._onDropHandler.bind(self),
                        onEnter: self.onEnterHandler.bind(self),
                        onLeave: self.onLeaveHandler.bind(self),
                        onCancel: self.onCancelHandler.bind(self)
                    };
            if (self.reflowable) {
                dragOptions.onDrag = function (el, e) {
                    el.setStyle('top', e.event.pageY);
                };
            }

            var drag = new Drag.Move(clone, dragOptions);
            drag.start(event);
            context.fireEvent('dragStarted', {'draggable': clone});
        };
    },
    setupDraggables: function () {
        this.options.draggables.addEvent('mousedown', this.startDragHandler(this));
        this.options.draggables.addEvent('touchstart', this.startDragHandler(this));
    },
    makeDraggable: function (draggable) {
        this.options.draggables.push(draggable);
        draggable.addEvent('mousedown', this.startDragHandler(this));
        draggable.addEvent('touchstart', this.startDragHandler(this));
    },
    removeDraggable: function (draggable) {
        Array.erase(this.options.draggables, draggable);
        draggable.removeEvents('mousedown');
        draggable.removeEvents('touchstart');
    },
    resetDummy: function (dummyDraggable) {
        dummyDraggable.destroy();
    },
    onCancelHandler: function (dragging) {
        this.resetDummy(dragging);
        this.fireEvent('dragEnded', {'draggable': dragging});
    },
    onEnterHandler: function (draggable, droppable) {
        droppable.addClass(this.options.hoverClass);
    },
    onLeaveHandler: function (draggable, droppable) {
        droppable.removeClass(this.options.hoverClass);
    },
    _onDropHandler: function (dummyDraggable, droppable) {
        // Manually calculate the dropped-on droppable if this is a reflowable.
        if (this.reflowable) {
            var abs = Math.abs;
            droppable = this.options.droppables.map(function (drop) {
                // Get sizing information for each droppable and the position of the draggable relative to it.
                var position = Object.merge(drop.getCoordinates(), dummyDraggable.getPosition(drop));
                return {
                    x: position.x,
                    y: position.y,
                    w: position.width,
                    h: position.height,
                    el: drop
                };
            }).filter(function (pos) {
                // Filter out droppables that are not intersected by the draggable.
                return abs(pos.x) < pos.w && abs(pos.y) < pos.h;
            }).reduce(function (best, cur) {
                // Use the current if we don't have a best choice.
                if (best.el === null)
                    return cur;
                // Determine whether the current is better than the currently selected best.
                return (abs(best.x) + abs(best.y) < abs(cur.x) + abs(cur.y)) ? best : cur;
            }, {el: null}).el;
        }

        this.fireEvent('dragEnded', {'draggable': dummyDraggable, 'droppable': droppable});
        this.onDropHandler(dummyDraggable.orig, droppable);
        this.resetDummy(dummyDraggable);
    },
    onDropHandler: function (dummyDraggable, droppable) {
    }
});
/* Main Game class */
md.dragdrop.MatchingGame = new Class({
    Extends: md.dragdrop.BaseGame,
    initialize: function (containerEl, options) {
        this.refactorDrag();
        this.parent(containerEl, options);
        // Create the results manager.
        this.resultManager = new md.dragdrop.ResultManager(this.options.draggables.length, this.options.result);
        // Update and show the finish message when all questions are correct.
        this.resultManager.addEvent('allQuestionsCorrect', function (data) {
            $$(this.options.finishMsgSelector).getElement('span').set('text', Math.round(data.correctPercentage) + '%');
            $$(this.options.finishMsgSelector).fade('in');
        }.bind(this));
        // Add our drag event handlers.
        this.addEvents({
            dragStarted: function (event) {
                // Get the original element being dragged.
                var draggable = event.draggable.orig;
                // Save a reference to the current display style.
                draggable.setAttribute('data-origDisplay', draggable.getStyle('display'));
                // Hide the original element.
                draggable.setStyle('display', 'none');
            },
            dragEnded: function (event) {
                // Restore the original display style (probably showing the element.)
                event.draggable.orig.setStyle('display', event.draggable.orig.getAttribute('data-origDisplay'));
            }
        });
    },
    refactorDrag: function () {
        Class.refactor(Drag, {attach: function () {
                this.handles.addEvent('touchstart', this.bound.start);
                return this.previous.apply(this, arguments);
            }, detach: function () {
                this.handles.removeEvent('touchstart', this.bound.start);
                return this.previous.apply(this, arguments);
            }, start: function (event) {
                document.body.addEvents({touchmove: this.touchmoveCheck = function (event) {
                        event.preventDefault();
                        this.bound.check(event);
                    }.bind(this), touchend: this.bound.cancel});
                this.previous.apply(this, arguments);
            }, check: function (event) {
                if (this.options.preventDefault)
                    event.preventDefault();
                var distance = Math.round(Math.sqrt(Math.pow(event.page.x - this.mouse.start.x, 2) + Math.pow(event.page.y - this.mouse.start.y, 2)));
                if (distance > this.options.snap) {
                    this.cancel();
                    this.document.addEvents({mousemove: this.bound.drag, mouseup: this.bound.stop});
                    document.body.addEvents({touchmove: this.bound.drag, touchend: this.bound.stop});
                    this.fireEvent('start', [this.element, event]).fireEvent('snap', this.element);
                }
            }, cancel: function (event) {
                document.body.removeEvents({touchmove: this.touchmoveCheck, touchend: this.bound.cancel});
                return this.previous.apply(this, arguments);
            }, stop: function (event) {
                document.body.removeEvents({touchmove: this.bound.drag, touchend: this.bound.stop});
                return this.previous.apply(this, arguments);
            }});
    },
    onDropHandler: function (draggable, droppable) {
        // If we have a droppable.
        if (droppable) {
            // Determine whether this drop is correct or not.
            var correct = false;
            if (this.options.isCorrectCheck !== undefined) {
                // If we have a validation function, let it determine whether this is a correct drop.
                correct = this.options.isCorrectCheck(draggable, droppable);
            } else {
                // Otherwise, the drop is correct if the draggable and droppable's data-textid properties match.
                correct = draggable.get('data-textid') != undefined && draggable.get('data-textid') === droppable.get('data-textid');
            }

            // If this was a correct drop
            if (correct) {
                // Get rid of the draggable clone.
                draggable.destroy();
                // Set the draggable and droppable validation classes to the correct state.
                this.options.droppables.removeClass(this.options.incorrectClass);
                droppable.addClass(this.options.correctClass);
                draggable.removeClass(this.options.incorrectClass);
                // Make this droppable no longer droppable.
                this.options.droppables.erase(droppable);
                // Play the correct sound if present.
                if (droppable.getElement('span') !== null && droppable.getElement('span').hasAttribute('data-ibooks-audio-src')) {
                    var wordAudio = new Audio(droppable.getElement('span').getAttribute('data-ibooks-audio-src'));
                    wordAudio.play();
                }

                // Update the results manager.
                this.resultManager.incrementCorrect();
                // If we have a correct handler, call it.
                if (this.options.correctHandler) {
                    this.options.correctHandler(draggable, droppable);
                }
                // If this was an incorrect drop.
            } else {
                // Set draggable and droppable validation classes to the incorrect state.
                draggable.addClass(this.options.incorrectClass);
                droppable.addClass(this.options.incorrectClass);
                // Play the incorrect sound if present.
                if (this.errorAudio) {
                    this.errorAudio.play();
                }

                // Update the results manager.
                this.resultManager.incrementIncorrect();
                // Call the incorrect handler if present.
                if (this.options.incorrectHandler) {
                    this.options.incorrectHandler(draggable, droppable);
                }
            }

            // Call the Base Game's leave handler.
            this.onLeaveHandler(draggable, droppable);
        }
    }
});
md.dragdrop.CollectionGame = new Class({
    Extends: md.dragdrop.BaseGame,
    initialize: function (containerEl, options) {
        this.parent(containerEl, options);
        // Setup state variables.
        this.dropCount = 0;
        // Add our drag event handlers.
        this.addEvents({
            dragStarted: function (event) {
                event.draggable.addClass('drag');
                event.draggable.orig.addClass('active');
            },
            dragEnded: function (event) {
                event.draggable.orig.removeClass('active');
            }
        });
    },
    onDropHandler: function (draggable, droppable) {
        // If we have a drop taget and have not reached our drop limit.
        if (droppable && this.options.dropLimit > this.dropCount) {
            // Determine whether the drop is correct.
            var correct = false;
            if (this.options.isCorrectCheck !== undefined) {
                // Let the validation function determine if we have one.
                correct = this.options.isCorrectCheck(draggable, droppable);
            } else {
                // Otherwise, the drop is correct if the draggable and droppable's data-textid properties match.
                correct = draggable.get('data-textid') !== undefined && draggable.get('data-textid') === droppable.get('data-textid');
            }

            // If the drop is correct.
            if (correct) {
                // Call the correct handler if we have one.
                if (this.options.correctHandler) {
                    var boundCorrectHandler = this.options.correctHandler.bind(this);
                    boundCorrectHandler(draggable, droppable);
                }

                // Update the drop count.
                this.dropCount++;
            }

            // Call the Base Game's leave handler.
            this.onLeaveHandler(draggable, droppable);
            // Otherwise, if we have reached the drop limit.
        } else if (this.options.dropLimit <= this.dropCount) {
            window.alert('House is full (' + this.dropCount + ' members). Time for somebody to sleep outside.');
        }
    }
});
md.dragdrop.Quiz = new Class({
    Implements: [Options, Events],
    options: {
        reverse: false,
        minHeight: '100px',
        score: false,
        instructions: 'Drag the answers in the right column to the corresponding items on the left.',
        shuffleQuestions: false
    },
    initialize: function (el, questionSet, options) {
        this.el = el;
        // Setup state variables.
        var questions = [];
        var answers = [];
        this.questionMapping = {};
        this.reflowable = document.body.hasClass('book-type-reflowable') && !(window.parent && window.parent.Lindgren);
        // Merge the options.
        this.setOptions(options);
        this.reflowable ? this.el.getElement('.draggable-container') : $$('body')[0];
        // Prepare the questions.
        questionSet.each(function (question) {
            // Switch question and answer if the reverse option is set.
            if (this.options.reverse) {
                var q = question.answer;
                var a = question.question;
            } else {
                var q = question.question;
                var a = question.answer;
            }

            // Add the questions and answers to our state variables.
            questions.push(q);
            answers.push(a);
            this.questionMapping[q] = a;
        }.bind(this));
        // If we have duplicate questions, show error
        if (questions.length !== questions.unique().length) {
            throw('Questions has duplicate question, There should not be any duplicate questions');
        }

        if (this.options.shuffleQuestions) {
            // Shuffle and save the prepared questions.
            this.questions = questions.shuffle();
            // Save the prepared answers.
            this.answers = answers;
        } else {
            // Save the prepared questions.
            this.questions = questions;
            // Shuffle and save the prepared answers.
            this.answers = answers.shuffle();
        }

        // Create the UI.
        this.createDOM();
        // Add the drag interaction.
        this.makeDrag();
        if (this.options.score) {
            this.incorrectAnswers = 0;
            this.correctAnswers = 0;
        }
    },
    makeDrag: function () {
        var matchingGameOptions = {
            draggables: this.el.getElements('.draggable'),
            droppables: this.el.getElements('.droppable'),
            draggablesContainer: this.reflowable ? this.el.getElement('.draggable-container') : $$('body')[0],
            correctHandler: function (draggable, droppable) {
                // Have the droppable accept the draggable.
                droppable.adopt(draggable.clone());
                // Cleanup the temporary draggable.
                draggable.destroy();
                if (this.options.score) {
                    this.correctAnswers += 1;
                    this.el.getElement('.dragdropquiz-correct-score').set('text', this.correctAnswers);
                }
                this.fireEvent("answerCorrect", {draggable: draggable, droppable: droppable, correctAnswers: this.correctAnswers});
            }.bind(this),
            incorrectHandler: function () {
                if (this.options.score) {
                    this.incorrectAnswers += 1;
                    this.el.getElement('.dragdropquiz-incorrect-score').set('text', this.incorrectAnswers);
                }
                this.fireEvent("answerIncorrect", {incorrectAnswers: this.incorrectAnswers});
            }.bind(this)
        };
        if (this.reflowable) {
            matchingGameOptions.appContainer = this.el.getElement('.dragdropquiz-container');
        }

        // Create a Matching Game to handle the drag and drop interaction.
        new md.dragdrop.MatchingGame(this.el, matchingGameOptions);
    },
    reset: function (e) {
        e.stop();
        // Reset the UI.
        this.createDOM();
        // Recreate the drag interaction.
        this.makeDrag();
        if (this.options.score) {
            this.incorrectAnswers = 0;
            this.correctAnswers = 0;
        }
    },
    createDOM: function () {
        var widgetContainer = new Element('div', {'class': 'dragdropquiz-container'}),
                instructions = new Element('div', {'class': 'dragdropquiz-instructions', text: this.options.instructions}),
                resetButtonWrapper = new Element('div', {'class': 'dragdropquiz-reset-wrapper'}),
                resetButton = new Element('button', {'class': 'dragdropquiz-reset btn', text: 'Reset', events: {click: this.reset.bind(this)}});
        widgetContainer.adopt(instructions);
        if (this.options.score) {
            var scoreContainer = new Element('div', {'class': 'dragdropquiz-score'}),
                    correctContainer = new Element('div', {html: '<span class="correct-label">Correct</span>'}),
                    // correctScore = new Element('blockquote').adopt(new Element('small', { 'class': 'dragdropquiz-correct-score', text: '0' })),
                    correctScore = new Element('div', {'class': 'dragdropquiz-correct-score', text: '0'}),
                    incorrectContainer = new Element('div', {html: '<span class="incorrect-label">Incorrect</span>'}),
                    // incorrectScore = new Element('blockquote').adopt(new Element('small', { 'class': 'dragdropquiz-incorrect-score', text: '0' }));,
                    incorrectScore = new Element('div', {'class': 'dragdropquiz-incorrect-score', text: '0'});
            widgetContainer.adopt(scoreContainer.adopt(correctContainer.adopt(correctScore), incorrectContainer.adopt(incorrectScore)));
        }


        this.questionContainer = new Element('div', {'class': 'draggable-container', styles: {'min-height': this.options.minHeight}});
        this.questions.each(function (question) {
            var questionWrapper = new Element("div", {"class": "question-wrapper"});
            questionWrapper.adopt(new Element('div', {
                'class': 'draggable question chaucer smartwidget',
                'data-textid': this.questionMapping[question],
                'text': question
            }));
            this.questionContainer.adopt(questionWrapper);
        }.bind(this));
        this.answerContainer = new Element('div', {'class': 'droppable-container'});
        this.answers.each(function (answer) {
            var answerText = new Element('div');
            answerText.adopt(new Element('h5', {text: answer}));
            var answerHolder = new Element('div', {
                'class': 'droppable',
                'data-textid': answer
            });
            var answerEl = new Element('div', {'class': 'answer'});
            answerEl.adopt([answerText, answerHolder]);
            this.answerContainer.adopt(answerEl);
        }.bind(this));
        resetButtonWrapper.adopt(resetButton);
        widgetContainer.adopt(this.answerContainer, this.questionContainer, resetButtonWrapper.adopt(resetButton));
        this.el.empty();
        this.el.adopt(widgetContainer);
        // If we are in a reflowable.
        if (this.reflowable) {
            // Make the container position relative.
            widgetContainer.setStyle('position', 'relative');
            // Add a page break class to the parent.
            this.el.getParent().addClass('page-break');
        }
    }
});
md.dragdrop.StepByStepQuiz = new Class({
    Extends: md.dragdrop.Quiz,
    initialize: function (el, questionSet, options) {
        this.parent(el, questionSet, options);
        this.hideQuestions();
        this.addEvent("answerCorrect", function (data) {
            this.hideQuestions();
        }.bind(this));
    },
    hideQuestions: function (visibleIndex) {
        // if(this.draggablesContainer == document.body) {
        // 	throw("StepByStepQuiz doesnt work with body as draggable container.")
        // }
        var draggables = this.questionContainer.getElements('.draggable');
        draggables.setStyle("display", "none");
        if (draggables.length > 0) {
            draggables[0].setStyle("display", "block");
        }
    },
    reset: function (e) {
        this.parent(e);
        this.hideQuestions();
    }
})
