/*
---
script: metrodigi-image-compare.js

description: Provides a filmstrip and two image displays which are used to compare two images at once.

requires: 
    - Core/DomReady
    - Core/Class

provides: [md.widgets.ImageCompare]
...
*/

var md = md || {};
md.widgets = md.widgets || {};
md.widgets.ImageCompare = new Class({
    Implements: [Options, Events],
    options: {
        featured : { left: null, right: null },
        images   : [],
        zoom     : false
    },
    initialize: function (el, options) {
        var makeDrag = function (e) {
            var clone = e.target.clone(),
                wrap  = new Element('div', { 'class': 'drag-clone' }).setStyles(Object.merge(e.target.getCoordinates(), { opacity: '0.6', position: 'absolute', display: 'block' })),
                drag;

            (this.reflowable ? this.el : document.body).adopt(wrap.adopt(clone));

            drag = new Drag.Move(wrap, {
                droppables: this.el.getElements('.drop-left, .drop-right'),
                onDrag: this.reflowable ? function (el, e) { e.stop(); el.setStyle('top', e.event.pageY); } : function (el, e) { e.stop(); },
                onDrop: function (el, drop) {
                    // Manually calculate the dropped-on droppable if this is a reflowable.
                    if (this.reflowable) {
                        var abs = Math.abs;

                        drop = this.el.getElements('.drop-left, .drop-right').map(function (drop) {
                            // Get sizing information for each droppable and the position of the draggable relative to it.
                            var position = Object.merge(drop.getCoordinates(), el.getPosition(drop));

                            return {
                                x: position.x,
                                y: position.y,
                                w: position.width,
                                h: position.height,
                                el: drop
                            };
                        }).filter(function (pos) {
                            // Filter out droppables that are not intersected by the draggable.
                            return abs(pos.x) < pos.w && abs(pos.y) < pos.h;
                        }).reduce(function (best, cur) {
                            // Use the current if we don't have a best choice.
                            if (best.el === null) return cur;

                            // Determine whether the current is better than the currently selected best.
                            return (abs(best.x) + abs(best.y) < abs(cur.x) + abs(cur.y)) ? best : cur;
                        }, { el: null }).el;
                    }

                    if (drop) {
                        if (this.options.zoom && !this.reflowable) {
                            this.makeImageMap(drop, el.getElement('img'));
                        } else {
                            drop.empty().adopt(el.getElement('img'));
                        }
                    }

                    el.destroy();
                    drag.detach();
                }.bind(this),
                onCancel: function (el) { el.destroy(); },
                onComplete: function (el) { el.destroy(); }
            });

            e.stop();
            drag.start(e);
        }.bind(this);

        this.reflowable = document.body.hasClass('book-type-reflowable') && !(window.parent && window.parent.Lindgren);
        if (this.reflowable) {
            this.refactorDrag();
        }

        this.el = el;
        this.setOptions(options);
        this.render();

        this.el.addEvent('touchstart:relay(.film-strip img)', makeDrag);
        this.el.addEvent('mousedown:relay(.film-strip img)', makeDrag);
    },
    defaultZoom: function (dropCoor, img) {
        var defaultZoom;

        if (dropCoor.width < dropCoor.height) {
            // Ratio applied to width, zoom level applied to height.
            if (img.naturalWidth > img.naturalHeight) {
                defaultZoom = dropCoor.width / (img.naturalWidth / img.naturalHeight);
            } else {
                defaultZoom = dropCoor.width / (img.naturalHeight / img.naturalWidth);
            }
        } else {
            // Ratio applied to height, zoom level applied to width.
            if (img.naturalWidth > img.naturalHeight) {
                defaultZoom = dropCoor.height / (img.naturalHeight / img.naturalWidth);
            } else {
                defaultZoom = dropCoor.height / (img.naturalWidth / img.naturalHeight);
            }
        }

        return defaultZoom;
    },
    refactorDrag: function () {
        // Make drag events use client instead of page coordinates in reflowables.
        Class.refactor(Drag, {
            start: function (e) {
                e.page = e.client;
                this.previous(e);
            },
            drag: function (e) {
                e.page = e.client;
                this.previous(e);
            }
        });
    },
    makeImageMap: function (container, img) {
        var coor        = container.getCoordinates(),
            imageMap    = new Element('div').setStyles(coor),
            defaultZoom = this.defaultZoom(coor, img);

        container.empty().adopt(imageMap);

        new md.widgets.ImageMap(imageMap, {
            imageUrl    : img.get('src'),
            defaultZoom : defaultZoom,
            maxZoom     : defaultZoom * 2
        });
    },
    render: function () {
        var elCoors = this.el.getCoordinates();

        // If the element's width or height are less than the minimum dimensions, then the widget is not
        // safe to render yet. We terminate this call early and call the function again on the next turn
        // of the event loop.
        if (elCoors.width < 100 || elCoors.height < 100) {
            setTimeout(this.render.bind(this), 0);
            return;
        }

        var dropStyle        = { width: Math.floor(elCoors.width * 0.45), height: Math.floor(elCoors.height * 0.47) },
            leftBox          = new Element('div', { 'class': 'drop-left', text: 'Drag an image here to compare.', styles: dropStyle }),
            rightBox         = new Element('div', { 'class': 'drop-right', text: 'Drag an image here to compare.', styles: dropStyle }),
            filmStripWrapper = new Element('div', { 'class': 'film-strip-wrapper' }),
            filmStrip        = new Element('div', { 'class': 'film-strip' }),
            imgs             = this.options.images.map(function (img) { return new Element('img', { src: img.url }); });

        filmStripWrapper.adopt(filmStrip.adopt(imgs));
        this.el.empty().adopt(leftBox, rightBox, filmStripWrapper);

        // Add the starting images if they are present.
        if (this.options.featured.left) {
            var leftImg = new Element('img', { src: this.options.featured.left.url });
            leftBox.empty().adopt(leftImg);
            if (this.options.zoom && !this.reflowable) {
                leftImg.onload = function () { this.makeImageMap(leftBox, leftImg); }.bind(this);
            }
        }
        if (this.options.featured.right) {
            var rightImg = new Element('img', { src: this.options.featured.right.url });
            rightBox.empty().adopt(rightImg);
            if (this.options.zoom && !this.reflowable) {
                rightImg.onload = function () { this.makeImageMap(rightBox, rightImg); }.bind(this);
            }
        }

        this.el.setStyle('height', this.options.height);

        // If we are in a reflowable.
        if (this.reflowable) {
            this.el.parentNode.addClass('page-break');
        }
    }
});
