/*
---
script: metrodigi-media.js

description: Provides media (Audio / Video) related functions and objects 

requires: [Core/Core]

provides: [md.media]

...
*/
var md = md || {};
md.media = md.media || {};
md.media.AudioManager = new Class({
	Implements: [Options, Events],
	initialize:function(options){
		this.setOptions(options);
		this.map = {};
		if(this.options.audioUrlAttr) {
			var attr = '['+this.options.audioUrlAttr+']';
			Array.each($$(attr), function(imgEl){
			    var audioUrl = imgEl.getAttribute(this.options.audioUrlAttr);
				if(!this.hasAudio(audioUrl)) {
			    	this.addAudio(audioUrl);
			    	console.log('Audio registered. ', audioUrl);
				}
			}.bind(this));
		}
	},
	addAudio:function(url) {
		this.map[url] = new Audio(url);
		this.map[url].addEventListener('ended',function(){
			console.log('audio playing ended ' + url);
		});
		
	},
	hasAudio:function(url) {
		return this.map[url]!=undefined;
	},
	getAudio: function(url) {
		if(this.hasAudio === undefined) {
			throw('no audio found with url: ', url);
		}
		return this.map[url];
	},
	addAndGetAudio: function(url) {
		if(!this.hasAudio(url)) {
			this.addAudio(url);
		}
		return this.getAudio(url);
	},
	play:function(url, element) {
		var audio = this.map[url];
		
		if(this.isPlaying()) {
			if(this.playingAudio != audio ) {
				this.playingAudio.pause();
			} else {
				console.log('Not doing anything as Audio is already playing.');
				return;
			}
		}
		
		var self= this;
		var listenerFunction = function(){
			console.log('audio paused playing ' + url);
			self.fireEvent('audiostopped', {'url':url, 'audio':audio, 'element': element});
			audio.removeEventListener('pause', listenerFunction);
			self.playingAudio = undefined;
		};
		audio.addEventListener('pause',listenerFunction);
		// audio.addEventListener('timeupdate', function(e){	
		// 	console.log('time updated', e, audio.currentTime);
		// }.bind(this));
		console.log('playing ' + url);
		audio.play();
		this.playingAudio = audio;
	},
	isPlaying:function(url){
		if(url==undefined) {
			return this.playingAudio!=undefined && !this.playingAudio.paused;
		} else {
			if(this.playingAudio!=undefined) {
				return this.playingAudio == this.map[url];
			}
			return false;
		}
	},
	pause: function(url){
		var audio = this.map[url];
		audio.pause();
	},
	stop: function(url){
		var audio = this.map[url];
		audio.stop();
	}
	
});

md.media.AudioTextBinding = new Class({
	Implements: [Options, Events],
	options: {
		elAudioAttr: "data-audio",
		embedded: {
			elStartTimeAttr: "data-audio-start-time",
			elStopTimeAttr: "data-audio-stop-time",
			elDuration: "data-audio-duration",
			elSelector: '.audio-text'
		},
		activeClass: 'active',
		playedClass: 'played',
		embeddedConfig: false
	},
	dataList: [],
	initialize: function(container, audioManager, options){
		if(!container) {
			throw('container is not defined');
		}
		if(!audioManager) {
			throw('audioManager is not defined');
		}
		var audioUrl = container.getAttribute(this.options.elAudioAttr);
		if(!audioUrl) {
			throw(this.options.elAudioAttr + ' is compulsory attribute in Element');
		}
		this.container = container;
		this.setOptions(options);
		this.audio = audioManager.addAndGetAudio(audioUrl);
		this.audio.addEventListener('timeupdate', this.audioStep.bind(this));
		this.audio.addEventListener('ended', this.cleanup.bind(this));

		if(this.options.embeddedConfig) {
			this.setEmbeddedConfig();
		} else {
			this.setConfig();
		}
	}, 
	setEmbeddedConfig: function(){
		var els = this.container.getElements(this.options.embedded.elSelector);
		els.each(function(el) {
			var elData = {};
			elData.startTime= parseFloat(el.getAttribute(this.options.embedded.elStartTimeAttr));

			var elStopTime = el.getAttribute(this.options.embedded.elStopTimeAttr);
			var elDuration = el.getAttribute(this.options.embedded.elDuration);
			if(elStopTime) {
				elData.stopTime= parseFloat(elStopTime);
			} else {
				elData.duration= parseFloat(elDuration);
			}

			elData.active = false;
			elData.el = el;
			
			this.dataList.push(elData);
		}.bind(this));
	},
	setConfig: function(){
		if(!this.options.config) {
			throw('config option is not defined');
		}
		this.options.config.each(function(conf){
			var elData = conf;
			elData.active = false;
			elData.el = this.container.getElements(conf.selector);
			
			this.dataList.push(elData);
		}.bind(this));
	},
	audioStep: function(e){
		var time = this.audio.currentTime;
		this.dataList.each(function(elData) {
			var stopTime = elData.stopTime;
			if(stopTime == undefined) {
				stopTime = elData.startTime + elData.duration;
			}
			if(time > elData.startTime && time < stopTime) {
				if(elData.active === false) {
					elData.el.addClass(this.options.activeClass);
					// console.log('activated ', elData.el);
					elData.active=true;
				}
			} else {
				//IF Played Once
				if(elData.active === true) {
					elData.el.removeClass(this.options.activeClass);
					elData.el.addClass(this.options.playedClass);
					// console.log('deactivated ', elData.el);
					elData.active = false;
				}
			}
		}.bind(this))
	},
	cleanup: function() {
		console.log('cleaning up Elements');
		this.dataList.each(function(elData){
			elData.el.removeClass(this.options.activeClass);
			elData.el.removeClass(this.options.playedClass);
		}.bind(this));
	}
})

md.media.ui = md.media.ui || {};
md.media.ui.AudioControl = new Class({
	Implements: [Options, Events],
	options: {
		elAudioAttr: 'data-audio',
		playSelector: '.ac-play',
		pauseSelector: '.ac-stop',
	},
	initialize: function(container, audioManager, options){
		this.setOptions(options);
		this.container = container;
		
		if(!container) {
			throw('container is not defined');
		}
		if(!audioManager) {
			throw('audioManager is not defined');
		}
		this.audioUrl = container.getAttribute(this.options.elAudioAttr);
		if(!this.audioUrl) {
			throw(this.options.elAudioAttr + ' is compulsory attribute in Element');
		}
		
		this.audioManager = audioManager;
		this.audio = audioManager.addAndGetAudio(this.audioUrl);
		this.audio.addEventListener('ended',function(){
			this.resetControls();
		}.bind(this));
		
		this.playSelector = container.getElements(this.options.playSelector);
		this.pauseSelector = container.getElements(this.options.pauseSelector);
		
		this.resetControls();
		this.playSelector.addEvent('click', this.playHandler.bind(this));
		this.pauseSelector.addEvent('click', this.pauseHandler.bind(this));
	},
	resetControls: function(){
		this.pauseSelector.fade('hide');
		this.playSelector.fade('show');
	},
	playHandler: function(e){
		if(e) {
			e.stop();
		}
		this.audioManager.play(this.audioUrl);
		this.pauseSelector.fade('show');
		this.playSelector.fade('hide');
		this.playing = true;
	},
	pauseHandler: function(e){
		if(e) {
			e.stop();
		}
		this.audioManager.pause(this.audioUrl);
		this.pauseSelector.fade('hide');
		this.playSelector.fade('show');			
		this.playing = false;
	}
});

md.media.ui.AudioControlOverlay = new Class({
	Extends: md.media.ui.AudioControl,
	initialize: function(container, audioManager, animWrapper, options){
		this.parent(container, audioManager, options);
		this.wrapper = animWrapper;
		this.wrapper.addEvent('click', this.containerClicked.bind(this));
	},
	containerClicked: function(e){
		e.stop();
		if(this.playing) {
			this.pauseSelector.fade('in');
			setTimeout(function(){
				this.pauseSelector.fade('out');
			}.bind(this),600);
			
			//From this & parent pauseHandler();
			this.container.fade('show');
			setTimeout(function(){
				this.audioManager.pause(this.audioUrl);
				this.playSelector.fade('in');			
				this.playing = false;
			}.bind(this), 750);
		}
	},
	playHandler: function(e){
		this.parent(e);
		this.pauseSelector.fade('hide');		
		this.container.fade('out');
	},
	pauseHandler: function(e, delay){
		this.parent(e);
		this.container.fade('in');
	},
	resetControls: function(){
		this.parent();
		this.container.fade('in');
	},
	pause: function(){
		this.pauseHandler();
	}
});