/*
---
script: metrodigi-transform.js

description: Provides different custom HTML templating classes

requires: [Core/Core]

provides: [md.HtmlTransformer, md.HtmlTransformerEx, md.ModalContent, md.TemplateContent]

...
*/
var md = md || {};

/**
 	Base class which handles the content to content element transformation
 */
md.HtmlTransformer = new Class({
	Implements:[Options,Events],
	
	options: {
		// default clones the element before copy
	 	copy: true,
	
		// default strip BR tag from destination element during copy
		stripBR: true
	},
	
	initialize:function(srcEl, dest, mapping, options){
		this.setOptions(options);
		this.srcEl = srcEl;
		this.dest = dest;
		this.mapping = mapping;
		if(this.dest && this.dest.template) {
			this.fetchTemplate();
		}
	},
	
	fetchTemplate:function(){
		// For receiving template using XHR, due to CORS issue this is currently not in use
		// var myRequest = new Request({
		// 		    url: this.dest.template,
		// 		    method: 'get',
		// 		    onSuccess: function(responseText){
		// 				console.log('response received', responseText);
		// 				this.templateEl = Elements.from(responseText);
		// 				this.fireEvent('templateLoaded', {elements: this.templateEl});
		// 		    }.bind(this),
		// 		    onFailure: function(){
		// 				console.log('ERROR, Unable to load template');
		// 		    }
		// 		});
		// 		myRequest.send();
		this.templateEl = this.dest.template.el.clone().getChildren();
		
		this.fireEvent('templateLoaded', {elements: this.templateEl});
	},
	
	transform:function(){
		if(this.templateEl == undefined || this.srcEl == undefined || this.dest == undefined){
			return;
		}
			
		//If template and template is not loaded into templateEl yet.
		if(this.dest.template && this.templateEl === undefined) {
				var eventHandle = this.addEvent('templateLoaded', function(){
					this._transform();
					console.log('remove handle here', eventHandle);
				}.bind(this));
		} else {
			this._transform();
		}
	},
	
	_transform:function(){
		if(this.dest.template) {
			// Clear the destination element and copy the template to create basic structure
			this.dest.destinationEl.empty();
			this.dest.destinationEl.adopt(this.templateEl);
		}
		
		// Based on source-to-destination mapping, copy elements
		Object.each(this.mapping, function(destSelector, srcSelector) {
			this._replaceBlock(destSelector, srcSelector);
		}.bind(this));
	},
	
	_replaceBlock:function(destSelector, srcSelector){
		
		// creates array of elements to copy
		var srcElChilds = Array.map(this.srcEl.getElement(srcSelector).getChildren(), function(item){
			
			// if copy enabled, clone element
			if(this.options.copy) {
				item = item.clone();
			}
			
			// if BR not required, remove them
			if(this.options.stripBR) {
				
				// if element contains BR, collect and destroy them
				var brs = item.getElements('br');
				brs.destroy();
				
				// if element itself is BR, destroy it 
				if(item.match('br')) {
					item.destroy();
					return undefined;
				}
			}
			
			return item;
		}.bind(this));

		// copy generated elements to destination elemet
		this.dest.destinationEl.getElement(destSelector).adopt(srcElChilds);
	}
});

/**
 	Extension of basic transformation which handles 

  	1) template based transformation
  	2) disable popups in target element
 	3) block level BR tag removal
 */
md.HtmlTransformerEx = new Class({
	Extends : md.HtmlTransformer,
	
	_transform:function(){
		if(this.srcEl == undefined || this.dest == undefined){
			return;
		}
		
		if(this.dest.template) {
			this.dest.destinationEl.empty();
			this.dest.destinationEl.adopt(this.templateEl);
		}
		
		Object.each(this.options.properties, function(prop) {
			var srcBlockEl = this.srcEl.getElement(prop.srcClass).clone();

			// Disable popup trigger from destination element. This is to avoid opening another popup from popup.
			Array.each(srcBlockEl.getElements("a[data-trigger='ShowPopup']"), function(ele){
				var newEl = new Element('span');
				newEl.innerHTML = ele.innerHTML;
				newEl.replaces(ele);
			});
			
			if(prop['stripBR'] !== undefined && prop.stripBR == false){
				// if not to remove BR  
			}else{
				// remove BR
				if(this.options.stripBR || prop['stripBR'] ) {
					var brs = srcBlockEl.getElements('br');
					brs.destroy();
					if(srcBlockEl.match('br')) {
						srcBlockEl.destroy();
					}
				}
			}
			
			// replace destination template placeholder with generated HTML code
			this.dest.destinationEl.innerHTML = this.dest.destinationEl.innerHTML.replace("{{" + prop.placeholder + "}}", srcBlockEl.innerHTML);
		}.bind(this));
	},
});

/**
 	Wrapper for the basic html transformer. 
	It includes a default template and default mapping, so only source and target element is required for most cases.
	
	Example Use:
	
	new md.ModalContent($('source'), $('destination'), {}); 
 */
md.ModalContent = new Class({
	Implements: [Options],
	
	options: {
		// Default template to copy in destination element
		template: '<div><div class="modal-header"></div><div class="modal-body"></div><div class="modal-footer"><a class="close btn primaryb">Close</a></div></div>',
		
		// Default mapping to copy from source element to destination element
		mappings: {
			'.recipe-header':'.modal-header',
			'.recipe-body':'.modal-body'
		}
	},
	
	initialize: function(srcEl, desEl, options){
		this.setOptions(options);
		
		var transformer = new md.HtmlTransformer(srcEl, { 
			destinationEl: desEl, 
			template: {
				el: Elements.from(this.options.template)[0],
			}
		}, this.options.mappings, this.options);

		transformer.transform();
	}
});


/**
 	Wrapper for template based transformation. 
 	Requires source template and mapping for any transformation.
  
 	Example Use:
 
	var properties = [
		{ "srcClass":".recipe-header", "placeholder": "header", "stripBR": true },
		{ "srcClass":".recipe-body", "placeholder": "body" }
	];
	
 	new md.TemplateContent("#template_id", $('source'), $('target'), { 
		properties: properties, 
		stripBR: false // if omitted, will strip all BR by default
	});
 */
md.TemplateContent = new Class({
	Implements: [Options],
	
	options: {
		stripBR: true	// removes the BR tag from the destination copy
	},
	
	initialize: function(templateId, srcEl, desEl, options){
		this.setOptions(options);
		
		var templateEl = $$(templateId);
		if(templateEl == undefined || templateEl.length == 0){
			return;
		}
		
		var transformer = new md.HtmlTransformerEx(srcEl, { 
			destinationEl: desEl, 
			template: {
				el: $$(templateId)[0],
			}
		}, this.options.mappings, this.options);

		transformer.transform();
	}
});