/*
---
script: metrodigi-virtualreality.js

description: Provides CSS 3D based virtualreality 360. 

requires: 
 - Core/Class

provides: [md.widgets.VirtualRealityWidget]
...
*/
var md = md || {};
md.widgets = md.widgets || {};
md.widgets.VirtualRealityWidget = new Class({
    Implements:[Options, Events],
    options: {
        image: null,
        images: null,
        autoRotate: false,
        autoRotateSpeed: 6,
        fieldOfView: 75,
        verticalLimit: {
            bottom:-85,
            top:85
        }
    },
    initialize: function(el, options){
        this.el = $(el);
        this.setOptions(options);
        if(!this.options.images && !this.options.image) {
            throw("Expected images object with 6 properties (top, left, bottom, right, front, back) OR one paronamic image in options.");
        }
        
        this.setup();
    },
    toElement: function(){
        return this.el;
    },
    setup: function(){
        $(this).empty();
        var elWidth = $(this).getWidth();
        var elHeight = $(this).getHeight();

        var camera, scene, renderer;
        var geometry, material, mesh;
        var target = new THREE.Vector3();

        var lon = 90, lat = 0;
        var phi = 0, theta = 0;

        var touchX, touchY;

        //FOR Mac iBooks, As its not supporting event.webkitMovementX / Y
        var tmpX, tmpY;
        var exist = function(obj){
            return obj != undefined && obj != null
        }

        var onWindowResize = function() {
            camera.aspect = elWidth / elHeight;
            camera.updateProjectionMatrix();
            renderer.setSize( elWidth, elHeight );
        }

        var onDocumentMouseDown = function( event ) {
            event.preventDefault();
            document.addEventListener( 'mousemove', onDocumentMouseMove, false );
            document.addEventListener( 'mouseup', onDocumentMouseUp, false );
            tmpX = event.pageX;
            tmpY = event.pageY;
            //console.log("on mousedown = " + tmpX + "," + tmpY)
        }

        var onDocumentMouseMove = function( event ) {
            var movementX = event.movementX || event.mozMovementX || event.webkitMovementX || 0;
            var movementY = event.movementY || event.mozMovementY || event.webkitMovementY || 0;

            //change Value of MovementX & Y only if none of movement properties exist.
            if(!exist(event.movementX) && !exist(event.mozMovementX) && !exist(event.webkitMovementX)) {
                movementX = event.pageX-tmpX;
                movementY = event.pageY-tmpY;
                tmpX = event.pageX;
                tmpY = event.pageY;
            }
            lon -= movementX * 0.1;
            lat += movementY * 0.1;

            //console.log("on mousemove lon,lat= " + lon + "," + lat + "(" + movementX);
        }

        var onDocumentMouseUp = function( event ) {
            document.removeEventListener( 'mousemove', onDocumentMouseMove );
            document.removeEventListener( 'mouseup', onDocumentMouseUp );
        }

        var onDocumentMouseWheel = function( event ) {
            camera.fov -= event.wheelDeltaY * 0.05;
            camera.updateProjectionMatrix();
        }

        var onDocumentTouchStart = function( event ) {
            event.preventDefault();
            var touch = event.touches[ 0 ];
            touchX = touch.screenX;
            touchY = touch.screenY;
        }

        var onDocumentTouchMove = function( event ) {
            event.preventDefault();
            var touch = event.touches[ 0 ];
            lon -= ( touch.screenX - touchX ) * 0.1;
            lat += ( touch.screenY - touchY ) * 0.1;
            touchX = touch.screenX;
            touchY = touch.screenY;
        }

        var animate = function() {
            requestAnimationFrame( animate );
            if(this.options.autoRotate) {
                lon +=  this.options.autoRotateSpeed/100;    
            }
            lat = Math.max( this.options.verticalLimit.bottom, Math.min( this.options.verticalLimit.top , lat ) );
            phi = THREE.Math.degToRad( 90 - lat );
            theta = THREE.Math.degToRad( lon );
            target.x = Math.sin( phi ) * Math.cos( theta );
            target.y = Math.cos( phi );
            target.z = Math.sin( phi ) * Math.sin( theta );
            camera.lookAt( target );
            renderer.render( scene, camera );
        }.bind(this);


        var crop = function(side, callback){
            var img = new Image();
            img.onload = function(){
                var size = this.calcImageArea(img, side);
                console.log('Adding image ' + side.side, size);                

                var canvas = Elements.from('<canvas width="'+ (size.width) +'" height="'+ (size.height) +'"/>')[0];
                canvas.setStyle('display', 'none');//                                .hide()
                document.body.adopt(canvas);
                var ctx = canvas.getContext('2d');

                ctx.drawImage(img, size.x, size.y, size.width, size.height, 0, 0, size.width, size.height  );
                var base64ImageData = canvas.toDataURL();

                canvas.dispose();
                callback(side, base64ImageData)
            }.bind(this);
            img.src = side.url;
        }.bind(this);

        var initImages = function(){
            scene = new THREE.Scene();
            var isImages = this.options.images!=null;
            var sides = [                
                {
                    url: isImages?this.options.images.right:this.options.image,
                    position: [ -512, 0, 0 ],
                    rotation: [ 0, Math.PI / 2, 0 ],
                    side: "right"
                },
                {
                    url: isImages?this.options.images.left:this.options.image,
                    position: [ 512, 0, 0 ],
                    rotation: [ 0, -Math.PI / 2, 0 ],
                    side: "left"
                },
                {
                    url: isImages?this.options.images.top:this.options.image,
                    position: [ 0,  512, 0 ],
                    rotation: [ Math.PI / 2, 0, Math.PI ],
                    side: "top"
                },
                {
                    url: isImages?this.options.images.bottom:this.options.image,
                    position: [ 0, -512, 0 ],
                    rotation: [ - Math.PI / 2, 0, Math.PI ],
                    side: "bottom"
                },
                {
                    url: isImages?this.options.images.front:this.options.image,
                    position: [ 0, 0,  512 ],
                    rotation: [ 0, Math.PI, 0 ],
                    side: "front"
                },
                {
                    url: isImages?this.options.images.back:this.options.image,
                    position: [ 0, 0, -512 ],
                    rotation: [ 0, 0, 0 ],
                    side: "back"
                }
            ];

            for ( var i = 0; i < sides.length; i ++ ) {
                var side = sides[ i ];

                if(isImages) {
                    var img = document.createElement( 'img' );
                    img.width = 1026; // 2 pixels extra to close the gap.
                    var object = new THREE.CSS3DObject( img );
                    object.position.fromArray( side.position );
                    object.rotation.fromArray( side.rotation );
                    scene.add( object );
                    img.src = side.url;
                    img.setStyles({margin: 0, padding: 0});
                } else {
                    if(side.side == 'top' || side.side == 'bottom') {
                        continue;
                    }
                    crop(side, function(side, base64) {
                        var img = document.createElement( 'img' );
                        img.width = 1026; // 2 pixels extra to close the gap.
                        var object = new THREE.CSS3DObject( img );
                        object.position.fromArray( side.position );
                        object.rotation.fromArray( side.rotation );
                        scene.add( object );
                        img.src = base64;
                    });
                }
            }
        }.bind(this);


        var init = function() {
            camera = new THREE.PerspectiveCamera( this.options.fieldOfView, elWidth / elHeight, 1, 1000 );
            initImages();

            renderer = new THREE.CSS3DRenderer();
            renderer.setSize( elWidth, elHeight );
            $(this).adopt( renderer.domElement );


            document.addEventListener( 'mousedown', onDocumentMouseDown, false );
            //document.addEventListener( 'mousewheel', onDocumentMouseWheel, false );

            document.addEventListener( 'touchstart', onDocumentTouchStart, false );
            document.addEventListener( 'touchmove', onDocumentTouchMove, false );

            window.addEventListener( 'resize', onWindowResize, false );
        }.bind(this)

        init();
        animate();
    },
    calcImageArea: function(image, side){
        var leftMap = {'left':0, 'front':1, 'right':2, 'back':3};
        var cropSize = {};
        var size = { width: image.naturalWidth, height: image.naturalHeight };
        cropSize.width = size.width/4;
        cropSize.height = size.height;
        cropSize.x = leftMap[side.side]*cropSize.width;
        cropSize.y = (size.height - cropSize.height)/2;
        return cropSize;
    },
    _initSingleImage: function(){
        
    }
});