/*
---
script: metrodigi-widgets.js

description: Provides HTML widgets. 
             1> TextSizeWidget- Widget for changing size of Text

requires: 
 - Core/Core
 - Drag.Scroll

provides: [md.widgets.TextSizeWidget, md.widgets.Tabs, md.widgets.Timeline]
...
*/
var md= md || {};
md.widgets = md.widgets || {};
md.widgets.TextSizeWidget = new Class({
	INVALID_TAGS: ['AUDIO', 'VIDEO', 'IMG', 'BR', 'HR'],
	INVALID_CLASSES: ['chaucer-widget-item'],
	Implements:[Options],
	
	options: {
		target: null,
		defaultSize: 12,
		incrementPerc: 10,
		widgetContainerClass: 'text-zoom-controls', 
		widgetZoomInClass: 'zoom-in',
		widgetZoomOutClass: 'zoom-out',
		widgetZoomResetClass: 'zoom-reset',
		createEl: true
	},
	originalSize: undefined,
	initialize: function(ele, options){
		this.el = ele;
		this.setOptions(options);
		
		this.target = document.body;
		if(this.options.target) {
			this.target = this.options.target;
		};

		if(this.options.createEl) {
			var mainEl = new Element('ul', {'class': this.options.widgetContainerClass});
			var zoomInEl = new Element('li', {text: '+', 'class': this.options.widgetZoomInClass + " zoom-action"});
			var zoomOutEl = new Element('li', {text: '-', 'class': this.options.widgetZoomOutClass + " zoom-action"});
			var zoomResetEl = new Element('li', {text: '0', 'class': this.options.widgetZoomResetClass + " zoom-action"});
			var clearEl = new Element('div', {'style': 'clear: both'});
			mainEl.adopt(zoomInEl, zoomOutEl, zoomResetEl, clearEl);
			this.el.empty();
			this.el.adopt(mainEl);
		} else {
			mainEl = this.el.getElements('.'+this.options.widgetContainerClass);
			zoomInEl = this.el.getElements('.'+this.options.widgetZoomInClass);
			zoomOutEl = this.el.getElements('.'+this.options.widgetZoomOutClass);
			zoomResetEl = this.el.getElements('.'+this.options.widgetZoomResetClass);
		}

		zoomInEl.addEvent('click', function(e){
			e.preventDefault();
			this.changeFontSize(this.target, true);
		}.bind(this));
		
		zoomOutEl.addEvent('click', function(e){
			e.preventDefault();
			this.changeFontSize(this.target, false);
		}.bind(this));
		
		zoomResetEl.addEvent('click', function(e){
			e.preventDefault();
			this.resetFontSize(this.target);
		}.bind(this));
	},
	resetFontSize: function(textEl){
		var oriSize = textEl.getAttribute('data-ori-font-size');
		if(oriSize) {
			console.log('Changed to original size: ' + oriSize + ", el:", textEl);
			textEl.setStyle('font-size', oriSize);
		}
		textEl.getChildren().each(function(el) {
			this.resetFontSize(el);
		}.bind(this));
	},
	changeFontSize:function(textEl, increase){
		var isInvalid = this.INVALID_TAGS.contains(textEl.tagName);
		var isWidgetInvalid = this.INVALID_CLASSES.some(function(clazz){
			return textEl.hasClass(clazz) || textEl.getParent("." + clazz);
		});
		//CALCULATE FOR CHILD
		textEl.getChildren().each(function(el) {
			this.changeFontSize(el, increase);
		}.bind(this))

		if(isInvalid) {
			return;
		}

		var fontSize =  oriSize = textEl.getStyle('font-size');

		if(isWidgetInvalid) {
			textEl.setStyle('font-size', fontSize);
			return;
		}

		//Save original Size
		if(!textEl.hasAttribute('data-ori-font-size')) {
			textEl.setAttribute('data-ori-font-size', fontSize);
		}

		var unit = '%';
		if(fontSize.indexOf('%')>=0) {
			fontSize = fontSize.substr(0, fontSize.length - 1);
		} else {
			//every other unit is 2 char.
			unit = fontSize.substr(fontSize.length - 2, fontSize.length);
			fontSize = fontSize.substr(0, fontSize.length - 2);
		}

		fontSize = parseFloat(fontSize);
		var sizeChangeVal = fontSize*this.options.incrementPerc / 100;
		if(increase) {
			fontSize = fontSize + sizeChangeVal;	
		} else {
			fontSize = fontSize - sizeChangeVal;	
		}
		
		textEl.setStyle('font-size', fontSize + unit);
		console.log('changed size to ' + (fontSize + unit) + " from " + oriSize +  ", el:", textEl);
		
	}
});
md.widgets.Tabs = new Class({
	Implements: Events,
	initialize: function(tabs, contents, opt) {
		this.tabs = tabs;
		this.contents = contents;
		if(!opt) opt = {};
		this.css = opt.selectedClass || 'selected'; 
		this.select(this.tabs[0]);
		tabs.each(function(tab){
			tab.addEvent('click',function(e){
				this.select(tab);
				e.stop();
			}.bind(this));
		}.bind(this));
	},

	select: function(el) {
		this.tabs.removeClass(this.css);
		el.addClass(this.css);
		this.contents.setStyle('display','none');
		var content = this.contents[this.tabs.indexOf(el)];
		content.setStyle('display','block');
		this.fireEvent('change',[content,el]);
	}
});

md.widgets.Timeline = new Class({
	initialize: function(container){
		new Drag.Scroll(container, {
    		axis: {x: true, y: false}    
  		});	
	}
});

md.widgets.ImageMap = new Class({
	Implements: [Events, Options],
	options: {
		defaultZoom: 100,
		defaultX: 0,
		defaultY: 0,
		zoomStep: 10,
		minZoom: 0,
		maxZoom: 250,
		left: 0,
		top: 0,
		wrapperEl: null,
		imageEl: null,
		controlsEl: null,
		imageUrl: null,
		showZoomControls: true,
	},
	currentZoom: null,
	wrapper: null,
	image: null,
	controlsWrapper: null,
	initialize: function (el, options) {
		// We check the container size and abort if the container element is not the crop size.
		// We try to intialize again on the next turn of the event loop if this is the case.
		// This prevents the Scrollable Image from initializing with a collapsed height and being unusable.
		var containerSize = el.getSize();
        if (containerSize.x !== options.cropWidth || containerSize.y !== options.cropHeight) {
            setTimeout(function () { this.initialize.call(this, el, options); }.bind(this), 0);
            return;
        }

		this.el = el;
		this.setOptions(options);
		var needDomSetup = !options.wrapperEl;
		if(needDomSetup) {
			this.setupDom();
		}
		this.wrapper = this.options.wrapperEl;
		this.image = this.options.imageEl;
		this.controlsWrapper = this.options.controlsEl;
		this.addControls();

		if(!needDomSetup) {
			this.attachBehavior();
		}
	},
	attachBehavior: function(){
		this.setZoom(this.options.defaultZoom);
		this._resetDrag();
		this.wrapper.scrollTo(this.options.left, this.options.top);
	},
	setupDom: function() {
		// <div id="map_wrapper">
		// 	<img id="map_image" src="http://placehold.it/1350x1350" style=""></img>
		// </div>
		// <div id="controls-container">
		// </div>
		this.el.empty();
		this.options.wrapperEl = new Element('div', {'class':'imagemap-wrapper'});
		var size = this.el.getSize();
		this.options.wrapperEl.setStyles({
			'position': 'relative', 'overflow':'hidden',
			'width': size.x+"px", 'height':size.y+"px"});
		
		this.options.imageEl = new Element('img', {
			src: this.options.imageUrl
		});
		this.options.imageEl.addEvent('load', function(){
			this.attachBehavior();
			// Reset minzoom if the image and container are both "portrait" orientation.
			// Prevents zoom out to blank areas.  See CHAUC-1060.
			if (this.image.naturalWidth < this.image.naturalHeight && this.el.getSize().x < this.el.getSize().y) {
				this.options.minZoom = this.el.getSize().x / this.image.naturalWidth * this.image.naturalHeight;
			}
		}.bind(this));

		this.options.controlsEl = new Element('div');

		this.options.wrapperEl.adopt(this.options.imageEl);

		var widgetWrapperEl = new Element('div', {'class':'widget-imagemap-wrapper'});
		widgetWrapperEl.adopt(this.options.wrapperEl);
		widgetWrapperEl.adopt(this.options.controlsEl);
		this.el.adopt(widgetWrapperEl);
	},
	addControls: function(){

		var createControl = function(text, isPlus){
			var controlEl = new Element('div', {
				'class': 'map-control',
				'text':text
			});
			controlEl.addEvent('click', function(e){
				e.stop();
				var zoom = this.currentZoom;
				if(isPlus) {
					zoom = zoom + this.options.zoomStep;
				} else {
					zoom = zoom - this.options.zoomStep;
				}
				this.setZoom(zoom);
			}.bind(this));
			return controlEl;
		}.bind(this);

		this.controlsWrapper.addClass('map-controls-container');
		this.controlsWrapper.adopt(createControl('+', true));
		this.controlsWrapper.adopt(createControl('-', false));
		if(!this.options.showZoomControls) {
			this.controlsWrapper.setStyle('display', 'none');
		}
	},
	_resetZoom: function(){
		var useWidth = this.el.getSize().x < this.el.getSize().y;
		var zoomWidth = this.currentZoom;
		var zoomHeight = this.currentZoom;
		if(useWidth) {
			zoomWidth = zoomHeight * this.image.naturalWidth / this.image.naturalHeight;
		} else {
			zoomHeight = zoomWidth * this.image.naturalHeight / this.image.naturalWidth;
		}
		this.image.setStyle('width', zoomWidth + "px");
		this.image.setStyle('height', zoomHeight + "px");
	},
	setZoom: function(zoomLevel){
		if(zoomLevel > this.options.maxZoom) {
			this.currentZoom = this.options.maxZoom;
		} else if(zoomLevel < this.options.minZoom) {
			this.currentZoom = this.options.minZoom;
		} else {
			this.currentZoom = zoomLevel;
		}
		this._resetZoom();
	},
	_resetDrag: function(){
		if(this.drag) {
			this.drag.drag.detach();
		}
		this.drag = new Drag.Scroll(this.wrapper, {
    		axis: {x: true, y: true}  
  		});
  		window.drag = this.drag;
	}
});

