/**
 * Click-to-reveal with transparent images over a solid background:
 *
 * @reference: beebe-psh-6e/.../ch01-other_custom-08
 * @author: Aaron Melocik
 */

/**
 * ClickToReveal constructor takes:                                                     OPTIONAL PARAMETERS:
 * {container, widgetInstructions, backgroundImageFileName, widgetContent[{}, {}, etc.] [[, canvasWidth, canvasHeight]]}
 */
ClickToReveal = new Class({
    Implements: [Options, Events],
    // hidden: true,
    options: {
        title: 'Interactive',
    },
    initialize: function (container) {
        var _this = this;

        /**
         * DOM Element references
         */
        _this.containerElement = $$('.container');
        _this.widgetInstructionsElement = $$('.widget-instructions');
        _this.clickableSectionElement = $$('.clickable-section')[0];
        _this.imageListElement = $$('.clickable-image-list')[0];
        _this.imageListElementWidth = _this.imageListElement.getStyle('width');
        _this.imageListElementHeight = _this.imageListElement.getStyle('height');

        _this.imageListBackgroundElement = $$('.clickable-image-background')[0];
        _this.imageCanvasElement = $$('.clickable-image-canvas')[0];
        _this.revealSectionElement = $$('.reveal-section')[0];

        /**
         * Instance-related data
         */
        _this.container = container;
        _this.imageFolderPath = document.URL.slice(0, document.URL.lastIndexOf('/') + 1) + 'images/';
        _this.imageElements = [];
        _this.revealedArticleElements = [];
        _this.numberOfImageElements = 0;
        _this.visibileLayer;

        /**
         * Canvas-related data
         */
        _this.devicePixelRatio = window.devicePixelRatio;

        _this.canvasDimensions = {
            height: _this.imageListElementHeight,
            width: _this.imageListElementWidth
        };

        if (_this.container.canvasHeight) {
            _this.canvasDimensions.height = _this.container.canvasHeight;
        }
        if (_this.container.canvasWidth) {
            _this.canvasDimensions.width = _this.container.canvasWidth;
        }

        _this.ctx; // 2d canvas context
        _this.disableSmoothing = false; // defaults to false

        /**
         * Function invocations
         */
        _this.buildWidgetInstructions();
        _this.buildWidgetBody();
    },
    /**
     * Inject text from widgetInstructions into the widget header.
     */
    buildWidgetInstructions: function () {
        var _this = this;
        _this.widgetInstructionsElement.set('text', _this.container.widgetInstructions);
    },
    /**
     * Build layers of canvas elements
     * to enable clicking through transparent
     * sections of art assets.
     *
     * This code block should be easily abstracted.
     */
    buildCanvasLayer: function (imagePath) {
        var _this = this;

        /**
         * Warning in case of high- or low-density pixel screens.
         * No solution currently implemented.
         */
        if (!_this.devicePixelRatio) {
            console.log('WARNING: devicePixelRatio is ' + _this.devicePixelRatio + ' instead of "1". This may cause rendering problems.');
        }

        /**
         * Build new canvas El and
         * explicitly set the
         * HTML & CSS width and height of the
         * <canvas> element to prevent blurry
         * images.
         *
         * Unfortunately this is necessary.
         */
        var newCanvas = new Element('canvas');
        newCanvas.set('text', 'Sorry, your browser doesn\'t support the "canvas" element.');
        
        // NATIVE-STYLE
        newCanvas.setStyle('width', _this.canvasDimensions.width.toInt());
        newCanvas.setStyle('height', _this.canvasDimensions.height.toInt());
        newCanvas.setAttribute('width', _this.canvasDimensions.width.toInt());
        newCanvas.setAttribute('height', _this.canvasDimensions.height.toInt());

        // Check for completion of assignment of width and height, else fall back to Mootools
        if (newCanvas.getAttribute('width').toInt() !== _this.canvasDimensions.width.toInt()) {
            // MOOTOOLS-STYLE:
            console.log('WARNING: setAttribute did not appear to work. newCanvas.getAttribute("width") =', newCanvas.getAttribute("width"), '. Falling back to MooTools styling.');
            newCanvas.set('width', _this.canvasDimensions.width.toInt());
        }
        if (newCanvas.getAttribute('height').toInt() !== _this.canvasDimensions.height.toInt()) {
            // MOOTOOLS-STYLE:
            console.log('WARNING: setAttribute did not appear to work. newCanvas.getAttribute("height") =', newCanvas.getAttribute("height"), '. Falling back to MooTools styling.');
            newCanvas.set('height', _this.canvasDimensions.height.toInt());
        }

        /**
         * Set the 2d context and optionally
         * disable image smoothing (in case of blurry rendering)
         */
        newCanvas.ctx = newCanvas.getContext('2d');
        if (_this.disableSmoothing) {
            newCanvas.ctx.mozImageSmoothingEnabled = false;
            newCanvas.ctx.webkitImageSmoothingEnabled = false;
            newCanvas.ctx.msImageSmoothingEnabled = false;
            newCanvas.ctx.imageSmoothingEnabled = false;
        }

        /**
         * Create a new image from the file path
         * and draw on cavas once the image
         * is loaded in the browser.
         * crossOrigin = 'Anonymous' is to avoid cross-origin security errors from <canvas>'s drawImage() function.
         */
        var img = new Image();
        img.crossOrigin = "Anonymous";
        // img.src = imagePath; // minimum change from previous iteration
        img.onload = function () {
            newCanvas.ctx.drawImage(img, 0, 0, _this.canvasDimensions.width.toInt(), _this.canvasDimensions.height.toInt());
        };

        // Shim for Chaucer to ensure img src is downloaded after onload function is specified.
        setTimeout(function () {
            img.setAttribute('src', imagePath);

            // Check for completion of assignment of src, else fall back to Mootools
            if (img.getAttribute('src') !== imagePath) {
                // MOOTOOLS-STYLE:
                console.log('WARNING: setAttribute did not appear to work. img.src =', img.getAttribute("src"), '. Falling back to MooTools style.');
                img.set('src', imagePath);
            } // end if (MooTools fallback)
        }, 3000, img, imagePath); // end setTimeout()

        /**
         * Return the canvas with
         * the context still accessible
         * as [[canvas]].ctx
         */
        return newCanvas;
    }, // end buildCanvasLayer()
    /**
     * Build the widget body by layering canvas elements,
     * associating them with revealed articles,
     * and registering the transparency-detecting event
     * handler on the background image.
     */
    buildWidgetBody: function () {
        var _this = this;

        _this.imageListBackgroundElement.set('src', 'images/' + _this.container.backgroundImageFileName);

        /**
         * Build layers of <canvas> and text <span>s
         * and their associted article as follows:
         *  <li class="clickable-image">
         *    <canvas class="clickable-image-canvas">
         *    <span class="clickable-text">
         *  </li>
         *  <article class="slide hidden">
         *
         * Articles are hidden and revealed by CSS.
         * These imageListItems are accessible in _this.imageElements
         */
        _this.container.widgetContent.forEach(function (element, index, array) {
            var imageListItem = new Element('li.clickable-image').inject(_this.imageListElement);
            imageListItem.set('id', 'clickable-image-' + index);
            imageListItem.zIndexLayer = index * -1;
            imageListItem.setStyle('z-index', imageListItem.zIndexLayer);

            imageListItem.canvasLayer = _this.buildCanvasLayer(_this.imageFolderPath + _this.container.widgetContent[index].imageFileName);
            imageListItem.canvasLayer.addClass('clickable-image-canvas');
            imageListItem.canvasLayer.set('id', 'clickable-image-canvas-' + index);
            imageListItem.canvasLayer.inject(imageListItem);

            var textSpan = new Element('span.clickable-text').inject(imageListItem);
            textSpan.set('id', 'clickable-text-' + index);
            textSpan.set('text', _this.container.widgetContent[index].clickableText);

            imageListItem.revealArticle = new Element('article.slide.hidden').inject(_this.revealSectionElement);
            imageListItem.revealArticle.set('id', 'slide-' + index);
            imageListItem.revealArticle.set('html', element.revealedText);

            _this.imageElements.push(imageListItem);
        }); // end _this.container.widgetContent.forEach()

        _this.numberOfImageElements = _this.imageElements.length;
        _this.imageListBackgroundElement.setStyle('z-index', _this.numberOfImageElements + 2);

        /**
         * This adds an event listener to the background object
         * to identify the first layer with a non-transparent
         * pixel in the click location. No layers actually
         * respond to the click event (pointer-events: none in CSS).
         *
         * On a click, the event bubbles down the array of
         * imageElements. If a pixel is non-transparent, the
         * currently-visible imageElement is re-hidden and the
         * target imageElement (including <article>) is visible.
         */
        _this.imageListBackgroundElement.addEventListener('click', function (e) {
            _this.imageElements.forEach(function (element, index, array) {
                if (_this.getTransparency(e, element.canvasLayer) === 255) {
                    try {
                        if (_this.visibleLayer !== element) {
                            _this.visibleLayer.setStyle('z-index', _this.visibleLayer.zIndexLayer - _this.numberOfImageElements - 10);
                            _this.visibleLayer.revealArticle.addClass('hidden');

                            element.setStyle('z-index', element.zIndexLayer + _this.numberOfImageElements + 10);
                            element.revealArticle.removeClass('hidden');

                            _this.visibleLayer = element;
                            return true;
                        }
                    } catch (e) { // catch the first 'undefined' TypeError exception
                        element.setStyle('z-index', element.zIndexLayer + _this.numberOfImageElements + 10);
                        element.revealArticle.removeClass('hidden');
                        _this.visibleLayer = element;
                        return true;
                    } // end catch(e)
                } // end if (this element's pixel is not transparent)
            }); // end imageElements.forEach()
        }); // end addEventListener(click, getTransparency);
    }, // end buildWidgetBody()
    
    /**
     * Helper function to determine if the pixel at
     * the cursor location of the current element is
     * transparent or not.
     */
    getTransparency: function (e, canvasElement) {
        var clickedPixel = canvasElement.ctx.getImageData(e.layerX, e.layerY, 1, 1);
        var pixelTransparency = clickedPixel.data[3];
        return pixelTransparency;
    },
    isTouchDevice: function () {
        return !!('ontouchstart' in window);
    }
});
