/**
 * Image Drag-and-drop dev authoring:
 *
 * @author:     Renato Valencia
 */

TermsAndDefinitions = new Class({
    Implements: [Options, Events],
    Extends: MetrodigiWidget,
    options: {
        delay: 2000,
        top_score: 7,
        hasDefinition: false
    },
    terms: [],
    challenge_mode: false,
    removeText: function (message) {
        var _this = this;
        _this.options.data.terms.each(function (item, index) {
            if (message.data.id === item.id) {
                _this.options.data.terms.splice(index, 1);
                return false;
            }
        });

        _this.options.data.texts.each(function (item, index) {
            if (message.data.id === item.id) {
                _this.options.data.texts.splice(index, 1);
                return false;
            }
        });

        var all_objects = _this.canvas2.getObjects();
        all_objects.each(function (item) {
            if (item.type === 'group') {
                if (message.data.id === item.id) {
                    _this.canvas2.remove(item);
                    return false;
                }
            }
        });

        this.options.data = _this.options.data;
        this.saveRequest({method: 'mdData', options: {'json': this.options.data}});
    },
    events: function () {
        var _this = this;
        this.canvas2.on('object:modified', function (e) {
            var p = e.target;
            var all_objects = _this.canvas2.getObjects();
            _this.options.data.texts = [];
            _this.options.data.lines = [];
            all_objects.each(function (item) {
                if (item.type === 'group') {
                    var position = {};
                    var combined = {"id": item.id, "group": item}
                    _this.options.data.texts.push(combined);
                    position.left = item.left;
                    position.top = item.top;
                    _this.options.data.terms.each(function (_item, index) {
                        if (_item.id === item.id) {
                            _this.options.data.terms[index].position = [position.left, position.top];
                            return false;
                        }
                    });
                } else {
                    _this.options.data.lines.push(item);
                }
            });
            _this.saveRequest({method: 'mdData', options: {'json': _this.options.data}});
        });

        this.canvas2.on('object:selected', function (e) {
            if (e.target.type === 'line') {
                document.onkeydown = function (key) {
                    key.preventDefault();
                    if (key.keyCode === 46) {
                        _this.canvas2.remove(e.target);
                        _this.options.data.lines = [];
                        _this.canvas2.forEachObject(function (o) {
                            if (o.type === 'line') {
                                _this.options.data.lines.push(o);
                            }
                        });
                        _this.saveRequest({method: 'mdData', options: {'json': _this.options.data}});
                    }
                };
            }
        });

        this.options.data = _this.options.data;
    },
    saveText: function (message) {
        var _this = this;
        var objects = this.canvas2.getObjects();
        var new_item = true;
        var position = {};
        var group = null;
        objects.each(function (item) {
            if (item.id === message.data.id) {
                new_item = false;
                item._objects[0].setText(message.data.term);
                item._objects[1].setText(message.data.definition);
                var term_width = item._objects[0].width;
                var def_width = item._objects[1].width;
                if (term_width >= def_width) {
                    var width = term_width
                } else {
                    var width = def_width
                }
                item._objects[0].setLeft(-(width / 2));
                item._objects[1].setLeft(-(width / 2));
                item.setWidth(width);
                position.left = item.left;
                position.top = item.top;
                _this.canvas2.renderAll();
                _this.canvas2.setActiveObject(item);
                group = item;
                return false;
            }
        });

        if (new_item) {
            group = new fabric.Group();
            group.id = message.data.id;
            var term = new fabric.Text(message.data.term, {left: 40, top: 20, fontSize: 15, fontWeight: 'bold'});
            group.addWithUpdate(term);
            var definition = new fabric.Text(message.data.definition, {left: 40, top: 40, fontSize: 15});
            group.addWithUpdate(definition);
            _this.canvas2.add(group);
            _this.canvas2.forEachObject(function (o) {
                if (o.type === 'group') {
                    o.hasControls = false;
                }
            });
            var combined = {"id": message.data.id, "group": group};
            _this.options.data.texts.push(combined);
        } else {
            this.options.data.texts.each(function (item, index) {
                if (item.id === group.id) {
                    _this.options.data.texts[index].group = group;
                    return false;
                }
            });
        }

        var new_item = true;
        this.options.data.terms.each(function (item, index) {
            if (item.id === message.data.id) {
                new_item = false;
                var new_term = {term: message.data.term, definition: message.data.definition, position: [position.left, position.top], id: message.data.id};
                _this.options.data.terms[index] = new_term;
                return false;
            }
        });


        if (new_item) {
            var new_term = {term: message.data.term, definition: message.data.definition, position: [40, 20], id: message.data.id};
            _this.options.data.terms.push(new_term);
        }

        this.options.data = _this.options.data;
        this.saveRequest({method: 'mdData', options: {'json': this.options.data}});


    },
    customPostMessages: function (message) {
        var _this = this;
        switch (message.data.method) {
            case 'addLine':
                var random = Math.floor((Math.random() * 1000) + 1);
                var line = new fabric.Line([20, 20, 100, 100], {fill: 'black', stroke: 'black', strokeWidth: 1});
                this.canvas2.add(line);
                this.options.data.lines.push(line);
                this.saveRequest({method: 'mdData', options: {'json': this.options.data}});
                break;
            case 'addImage':
                $$('.draw-area').setStyle('background-image', 'url(' + message.data.image_name + ')');
                this.saveRequest({method: 'mdSetAttribute', options: {selector: '.draw-area', attribute: 'style', value: 'background-image: url(' + message.data.image_name + ');'}});
                break;
            case 'saveText':
                this.saveText(message);
                break;
            case 'removeText':
                this.removeText(message);
                break;
        }
    },
    generateDom: function () {
        _this = this;
        var item,
                term_list = new Element('ul.term-list').inject(this.canvas, 'after'),
                _this = this;

        if (typeof this.options.data.terms !== 'undefined') {
            for (var i = 0; i < this.options.data.terms.length; i++) {
                this.terms[i] = new TermsAndDefinitions_Term({
                    id: i,
                    el: new Element('li.droppable.list-item.list-item-' + i).inject(term_list),
                    term: this.options.data.terms[i].term,
                    hasDefinition: _this.hasDefinition,
                    definition: this.options.data.terms[i].definition,
                    position: this.options.data.terms[i].position,
                    width: this.options.data.terms[i].width
                });
            }
            ;
        }


        fabric.util.enlivenObjects(this.options.data.lines, function (objects) {
            objects.forEach(function (o) {
                _this.canvas2.add(o);
            });
        });

        if (this.isEditMode) {
            var groups = [];
            this.options.data.texts.each(function (item) {
                item.group.id = item.id;
                groups.push(item.group);
            });
            fabric.util.enlivenObjects(groups, function (objects) {
                objects.forEach(function (o) {
                    _this.canvas2.add(o);
                });
            });
            $$('.widget-footer').hide();
        }

        _this.canvas2.forEachObject(function (o) {
            if (o.type === 'group') {
                o.hasControls = false;
            }
            if (!_this.isEditMode) {
                if (o.type === 'line') {
                    o.selectable = false;
                }
            }
        });
        this.challenge_term_list = new Element('ul').inject(this.challenge_term);

    },
    initialize: function (options) {
        var has_definitions = $('MetrodigiWidget').get('data-def');
        if (has_definitions === 'true') {
            this.options.hasDefinition = true;
        } else {
            this.options.hasDefinition = false;
        }
        options.extra = ['data-def'];
        this.parent(options);
        window.addEvent("mousemove", function () {
            document.getElement("body").addClass("remove-outline");
        })
                .addEvent("keyup", function (e) {
                    if (e.key == "tab") {
                        document.getElement("body").removeClass("remove-outline");
                    }
                });
        var _this = this;
        this.setOptions(options);
        this.container = this.options.container;
        this.canvas = this.container.getElement('canvas');
        this.ctx = this.canvas.getContext('2d');
        this.canvas2 = new fabric.Canvas('myCanvas', {width: 672, height: 495});
        this.feedback = this.container.getElement('.feedback');
        this.challenge_term = this.container.getElement('.challenge-term');
        this.question_number = this.container.getElement('.widget-footer .question-number');
        this.final_feedback = {
            score: this.container.getElement('.header-feedback.score'),
            type: this.container.getElements('.header-feedback .type'),
            questions_right: this.container.getElements('.header-feedback .questions-right'),
            questions_percent: this.container.getElements('.header-feedback .questions-percent')
        };

        this.hasDefinition = this.options.hasDefinition;
        this.generateDom();

        this.container.getElements('.total-questions').set('html', this.terms.length);


        if (!this.hasDefinition) {
            $$(this.container.getElement('.start-definition-challenge')).addClass('hidden');
            $$(this.container.getElement('.start-term-challenge')).set('text', 'Start Challenge');
        }

        this.container.getElement('.start-term-challenge').addEvent('click', function () {
            document.querySelector('div.notes').setAttribute('tabindex', '0');
            document.querySelector('div.notes').focus();
            this.startChallenge('term');

            if (_this.isTouchDevice()) {
                $$('.note.term')[0].set('text', 'Challenge: Drag the circle to connect the correct answer.')
            }
        }.bind(this));
        this.container.getElement('.start-definition-challenge').addEvent('click', function () {

            document.querySelector('div.notes').setAttribute('tabindex', '0');
            document.querySelector('div.notes').focus();

            this.startChallenge('definition');
            if (_this.isTouchDevice()) {
                $$('.note.definition')[0].set('text', 'Challenge: Drag the circle to connect the correct answer.')
            }
        }.bind(this));
        this.container.getElement('.end-challenge').addEvent('click', function () {
            this.endChallenge();
        }.bind(this));

        this.draggable = this.container.getElement('.draggable');
        this.draggable_previous_pos = {
            x: parseInt(this.draggable.getStyle('left')),
            y: parseInt(this.draggable.getStyle('top'))
        };
        this.draggable.makeDraggable({
            onDrag: function () {
                _this.drawLine();
            },
            droppables: $$('.droppable'),
            onEnter: function (draggable, droppable) {
                droppable.addClass('hover');
            },
            onLeave: function (draggable, droppable) {
                _this.container.getElements('.droppable').removeClass('hover');
            },
            onDrop: function (draggable, droppable) {
                var delay = 0;
                if (droppable) {
                    _this.checkAnswer(draggable.get('data-term-id'), droppable.get('data-term-id'), droppable);
                    delay = _this.options.delay;
                }
                var mover = new Fx.Move(_this.draggable, {
                    offset: {x: _this.draggable_previous_pos.x, y: _this.draggable_previous_pos.y},
                    position: 'upperLeft',
                    relativeTo: $$('.draw-area')[0],
                    onComplete: function () {
                        _this.drawLine();
                    }
                });
                mover.start();
            }
        });
        this.container.ontouchmove = function () {
            event.preventDefault();
        }
        this.isTouching = false;
        this.container.getElements(".droppable")
                .addEvents({
                    "click": function () {
                        if (_this.container.hasClass("Accessibility")) {
                            _this.checkAnswer(_this.container.getElement("div.note.accessibility:not(.status)").get('data-term-id'), this.get('data-term-id'), this);
                        }
                    },
                    "touchend": function () {
                        if (_this.challenge_mode) {
                            _this.isTouching = true;
                            _this.checkAnswer(_this.container.getElement("div.note.accessibility:not(.status)").get('data-term-id'), this.get('data-term-id'), this);
                        }
                    }
                });

        if (this.isTouchDevice()) {
            setInterval(function () {
                if ($$('button.accessibility')[0])
                    $$('button.accessibility')[0].remove();
            }, 1000);
        }

        if (this.options.challenge_mode != null && this.options.challenge_mode) {
            var challengeType = this.options.challengeType ? this.options.challengeType : 'term';
            document.querySelector('div.notes').setAttribute('tabindex', '0');
            document.querySelector('div.notes').focus();
            this.startChallenge(challengeType);

            if (_this.isTouchDevice()) {
                if (challengeType === 'term') {
                    $$('.note.term')[0].set('text', 'Challenge: Drag the circle to connect the correct answer.')
                } else {
                    $$('.note.definition')[0].set('text', 'Challenge: Drag the circle to connect the correct answer.')
                }
            }
        }

        this.events();
    },
    startChallenge: function (type) {
        var challenge_terms = this.shuffle(this.terms),
                _this = this;
        this.challenge_mode = true;
        this.container.addClass('challenge').set('data-challenge', type);
        this.container.removeClass('challenge-complete');
        this.container.getElements('.widget-header .header-feedback').removeClass('visible');
        this.type = type;

        this.feedback.set('html', '');
        this.challenge_term_list.set('html', '');
        this.question_number.set('html', 1);

        this.container.getElement("div.note.accessibility.status").set("html", this.container.getElement(".widget-footer .questions").get("text") + ". " + (this.type === "term" ? "Term Challenge: " : "Definition Challenge: "));

        for (var i = 0; i < challenge_terms.length; i++) {
            new Element('li.' + type, {
                html: '<span>' + challenge_terms[i].options[type] + '</span>',
                'data-term-id': challenge_terms[i].options.id
            }).inject(this.challenge_term_list);
        }
        ;
        this.draggable.set('data-term-id', challenge_terms[0].options.id);
        this.changeNoteAccessibility();


        this.drawLine();

        this.correct_answers_on_first_try = 0;
        this.correct_on_first_try = true;

        this.timer = setInterval(function () {
            if (_this.challenge_mode) {
                _this.line.end = [
                    parseInt(_this.draggable.getStyle('left')),
                    parseInt(_this.draggable.getStyle('top')) + _this.draggable.getSize().y / 2
                ];
                _this.render();
            }
        }, 33);
        if (!this.container.hasClass("Accessibility")) {
            if (this.container.getElement('button.accessibility')) {
                this.container.getElement('button.accessibility').focus();
            }
        }
    },
    endChallenge: function () {
        clearTimeout(this.timer);
        this.ctx.clearRect(0, 0, 700, 500);
        this.container.removeClass('challenge').removeProperty('data-challenge');
        this.container.getElements('.answered').removeClass('answered');
    },
    checkAnswer: function (answer1, answer2, element) {
        var _this = this,
                accessibility_on = _this.container.hasClass("Accessibility"),
                offset = accessibility_on ? 750 : 0;

        if (this.isTouching) {
            accessibility_on = true;
        }

        if (answer1 == answer2) {

            this.feedback
                    .set('html', message.correct)
                    .removeClass('correct').removeClass('incorrect')
                    .addClass('correct');
            element
                    .set("tabindex", -1)
                    .addClass('answered');

            if (this.correct_on_first_try) {
                this.correct_answers_on_first_try++;
            }
            setTimeout(function () {
                _this.challenge_term_list.children[0].destroy();
                if (_this.challenge_term_list.children.length > 0) {
                    _this.question_number.set('html', _this.terms.length - _this.challenge_term_list.children.length + 1);
                    _this.draggable.set('data-term-id', _this.challenge_term_list.children[0].get('data-term-id'));
                    if (accessibility_on) {
                        _this.container.getElement("div.note.accessibility.status").set("html", _this.container.getElement(".widget-footer .questions").get("text") + ". " + (_this.type === "term" ? "Term Challenge: " : "Definition Challenge: "));
                        _this.changeNoteAccessibility();
                        _this.container.getElement("div.instruction.accessibility").set("html", _this.container.getElement("div.instruction.accessibility").get("text") + " ");
                        _this.container.getElement("div.note.accessibility:not(.status)").focus();
                        _this.isTouching = false;
                    }
                }
                else {
                    _this.container.addClass('challenge-complete');

                    // Show score
                    var score = new Score(_this.terms.length, _this.correct_answers_on_first_try);
                    _this.final_feedback.score.set('html', score.getMessage());
                    _this.final_feedback.score.addClass('visible');

                    _this.feedback.removeProperty("aria-live");

                    _this.endChallenge();
                }
            }, _this.options.delay + offset - 10);
            this.correct_on_first_try = true;
        }
        else {
            this.correct_on_first_try = false;
            this.feedback
                    .set('html', message.incorrect)
                    .removeClass('correct').removeClass('incorrect')
                    .addClass('incorrect');
        }

        setTimeout(function () {
            document.querySelector('.shout').innerText = document.querySelector('[role="definition"].note.accessibility').innerText;
        }, 2000)

        setTimeout(function () {
            _this.feedback.set('html', '');
        }, _this.options.delay + offset);
    },
    shuffle: function (o) {
        for (var j, x, i = o.length; i; j = Math.floor(Math.random() * i), x = o[--i], o[i] = o[j], o[j] = x)
            ;
        return o;
    },
    drawLine: function () {
        this.line = {
            start: [
                this.challenge_term.getSize().x - 1,
                this.challenge_term.getSize().y / 2],
            end: [
                parseInt(this.draggable.getStyle('left')),
                parseInt(this.draggable.getStyle('top')) + this.draggable.getSize().y / 2]
        };
        this.ctx.clearRect(0, 0, 700, 500);
        this.ctx.beginPath();

        if (!this.container.hasClass("Accessibility")) {
            this.ctx.moveTo(this.line.start[0], this.line.start[1]);
            this.ctx.lineTo(this.line.end[0], this.line.end[1]);
        }

        this.ctx.stroke();
    },
    changeNoteAccessibility: function () {
        var element = this.challenge_term_list.children[0],
                text = element.getElement("span").get("html");
        this.container.getElement("div.note.accessibility:not(.status)").set({
            html: "<strong>" + (this.type === "term" ? "Term Challenge: " : "Definition Challenge: ") + "</strong>" + text,
            "data-term-id": element.get("data-term-id")
        });
    },
    isTouchDevice: function () {
        return typeof window.ontouchstart !== 'undefined';
    }
});

TermsAndDefinitions_Term = new Class({
    Implements: [Options, Events],
    options: {
    },
    initialize: function (options) {
        this.setOptions(options);
        this.el = this.options.el;
        this.position = this.options.position;

        new Element('span.term', {
            html: this.options.term
        }).inject(this.el);

        if (!this.options.hasDefinition) {
            new Element('span.definition.no-definition', {
                html: ''
            }).inject(this.el);
        } else {
            new Element('span.definition', {
                html: this.options.definition
            }).inject(this.el);
        }


        this.el.set('data-term-id', this.options.id);

        this.render();
    },
    render: function () {
        this.el.setStyles({
            left: this.options.position[0],
            top: this.options.position[1],
            width: this.options.width
        });
    }
});