/**
 * @class MetrodigiWidgetEditable
 *
 * This class contains all the necesary methods to enable
 * the widget editable functionality. For development class
 * MetrodigiWidget should extend from this class to inheritate
 * these methods. For production this can be skipped.
 *
 */
MetrodigiWidgetEditable = new Class({
    Implements: [Options, Events],
    root: "/Pearson/pearson-portal/",
    isEditMode: false,
    assets: [],
    /*
     * Initializes the editable widget
     * @param {Array} options
     */
    initialize: function (options) {
        var _this = this;

        // Merge the default options of the Class with the options passed in
        this.setOptions(options);

        // Parse URL parameters
        this.isEditMode = this.getURLParameter('mdedit') ? true : false;
        this.root = this.getURLParameter('docRoot') || this.root;

        // Enable edit mode
        if (this.isEditMode) {
            // set widget id
            var widget_id = _this.getURLParameter('widget_id');
            if (widget_id) {
                _this.widget_id = widget_id;
            }

            // Load the necessary assets to enable edit mode
            this.loadAssets(this.assets, function () {
                // Listen for messages from the portal
                _this.listenPostMessages();

                // Get widget data
                _this.sendPostMessages('get-widget-data');

                // Parse reusable editable elements
                _this.parseElements();

                // Enable custom edit mode funtionality
                _this.bindEditMode();

                // set basic md-edit-mode class
                $$('body')[0].addClass('md-edit-mode');


            });

            if (typeof options.extra !== 'undefined') {
                this.sendWidgetOptions(options.extra);
            } else {
                this.sendWidgetOptions();
            }
        }
        else {

            this.loadAssets(this.assets, function () {
                _this.listenPostMessages();
                _this.afterAssetsLoad();
            });
        }
        this.accessibility();
    },

    /*
     *
     * Event fired after assets were loaded and is non-edit mode
     */
    afterAssetsLoad: function () {
    },

    /*
     *
     * Reads accessibility_data json and adds it to the elements of  the DOM
     */
    accessibility: function () {
        var _this = this;
        window.addEvent('load', function () {
            if (typeof accessibility_data !== 'undefined') {
                // Images
                if (typeof accessibility_data.images !== 'undefined') {
                    var images = $$('img');
                    Array.each(accessibility_data.images, function (item) {
                        Array.each(images, function(image){
                            var src = image.get('src');
                            var n = src.lastIndexOf('/');
                            var final_src = src.substring(n + 1);
                            if (final_src === item.src) {
                                image.set('alt', item.alt);
                                return false;
                            }
                        });
                    });
                }
            }
            if (_this.isEditMode) {
                _this.sendPostMessages('access-ready');
            }
        });
    },
    /*
     * Gets a specified parameter from the URL
     * @param {String} name
     */
    getURLParameter: function (name) {
        name = name.replace(/[\[]/, "\\\[").replace(/[\]]/, "\\\]");
        var regex = new RegExp("[\\?&]" + name + "=([^&#]*)");
        var results = regex.exec(window.location.href);

        if (results == null)
            return null;
        else
            return results[1];
    },
    /*
     * Searches for reusable editable elements and
     * parses them
     */
    parseElements: function () {
        var _this = this;

        // Start intro-line element
        this.saveTimer = null;
        if ($$('.MetrodigiWidget #intro-line').length > 0) {
            $$('.MetrodigiWidget #intro-line')[0]
                .set('contenteditable', 'true')
                .set('data-md-editable', 'md-content-editable')
                .addEvent('input', function (e) {
                    clearTimeout(_this.saveTimer);
                    _this.saveTimer = setTimeout(function () {
                        _this.saveRequest({
                            method: 'setHTML',
                            selector: '#intro-line',
                            html: $('intro-line').get('html')
                        });
                    }, 1000);
                });
            setTimeout(function () {
                parent.postMessage({
                    method: 'toggle-checkbox',
                    selector: '#intro-line-input'
                }, "*");
            }, 700);
        }
        // End intro-line element

        // Define reusable editable elements
        // ...


        // Parse custom editable elements
        this.parseCustomElements();
    },
    /*
     * Searches for reusable editable elements and
     * parses them
     */
    parseCustomElements: function () {
        /*
         * This is just a function specification,
         * each widget instance has to redefine this function.
         */
    },
    /*
     * Loads dinamically the required assets for this element
     * @param {Array} assets
     * @param {Function} callback
     */
    loadAssets: function (assets, callback) {
        callback = callback || function () {
        };

        var imagesArray = new Array();
        var total = assets.length;
        var cont = 0;
        var onLoad = function () {
            if (++cont == total)
                callback();
        };

        if (total) {
            assets.each(function (e, i) {
                if (typeof (e) == "string")
                    e = {source: e};

                if (e.source.indexOf(".png") > 0 || e.source.indexOf(".gif") > 0 || e.source.indexOf(".jpg") > 0 || e.source.indexOf(".jpeg") > 0) {
                    imagesArray.push(e.source);
                }

                if (e.source.indexOf(".js") > 0) {
                    var extra = e.extra || {};
                    extra.onLoad = onLoad;
                    Asset.javascript(e.source, extra);
                }

                if (e.source.indexOf(".css") > 0) {
                    var extra = e.extra || {};
                    extra.onLoad = onLoad;
                    Asset.css(e.source, extra);
                }
            });

            if (imagesArray.length > 0) {
                Asset.images(imagesArray, {
                    onLoad: onLoad
                });
            }
        } else {
            callback();
        }
    },
    /*
     * Listens to messages sent from the portal
     */
    listenPostMessages: function () {
        _this = this,
            window.addEventListener("message", function (message) {
                switch (message.data.method) {
                    case "get-widget-data":
                        _this.widget_id = message.data.widget_id;
                        break;
                    case "change-display-mode":
                        $$(message.data.element).set("md-data-display-mode", message.data.mode);
                        break;
                    case "toggleHTML":
                        var prepend = false;
                        if (typeof message.data.prepend !== 'undefined') {
                            prepend = message.data.prepend;
                        }
                        _this.saveRequest({
                            method: "toggleHTML",
                            selector: message.data.wrapper,
                            elementID: message.data.elementID,
                            toggle: "" + message.data.value,
                            prepend: prepend
                        }, function () {
                            document.location.reload();
                        });
                        break;
                    default:
                        console.warn('Unknown md-widget message', message);
                }

                _this.customPostMessages(message);
            });
    },
    /*
     * Defines new custom cases for messages sent from the portal
     */
    customPostMessages: function (message) {
        /*
         * This is just a function specification,
         * each widget instance has to redefine this function.
         */
    },
    /*
     * Sends post message to the portal to notify that the widget is focused
     */
    notifyFocusCapture: function () {
        if (parent && parent != window) {
            this.sendPostMessages('widget-capture-focus');
        }
    },
    /*
     * Send widget options defined in the HTML
     */
    sendWidgetOptions: function (extra) {
        if (parent && parent != window) {
            var _this = this;
            this.readRequest(function (data) {
                var options = {
                    'data-xml-type': _this.container.get('data-xml-type'),
                    'widget-data': data
                };
                if (typeof extra !== 'undefined') {
                    Array.each(extra, function (item) {
                        options[item] = _this.container.get(item);
                    });
                }
                _this.sendPostMessages('widget-options', {
                    options: options
                });
                ;
            });
        }
    },
    /*
     * Sends message to the portal
     * @param {String} method
     * @param {Array} options
     */
    sendPostMessages: function (method, options) {
        params = {method: method}
        for (var attrname in options) {
            params[attrname] = options[attrname];
        }

        parent.postMessage(params, "*");
    },
    /*
     * Saves changes to the html file
     * @param {Object} request
     * @param {Function} callback
     */
    saveRequest: function (request, callback) {
        callback = callback || function () {
        };
        request.path = '';

        if (typeof request !== "string") {
            request = JSON.stringify(request);
        }
        var formData = new FormData();
        formData.append('savedata', request);
        formData.append('widget_id', this.widget_id);
        var xhr = new XMLHttpRequest();

        xhr.open('POST', this.root + 'api/index.php/save/');
        xhr.onload = function () {
            if (xhr.status === 200) {
                var response = JSON.parse(xhr.responseText);
                callback(response);
            } else {
                console.error(xhr.status, xhr.responseText);
            }
        };
        xhr.send(formData);
    },
    /*
     * Reads the json data from the selector #md-widget-data
     * @param {Function} callback
     */

    readRequest: function (callback) {
        callback = callback || function () {
        };
        var xhr = new XMLHttpRequest();
        xhr.onload = function () {
            if (xhr.status === 200) {
                var response = JSON.parse(xhr.responseText);
                callback(response);
            }
        };
        xhr.open('GET', this.root + 'api/index.php/save?widget_id=' + this.widget_id);
        xhr.send(null);
    },
    /*
     * Enables custom edit mode funtionality
     */
    bindEditMode: function () {
        /*
         * This is just a function specification,
         * each widget instance has to redefine this function.
         */
    }
});




/**
 * @class MetrodigiWidget
 *
 * This class contains all the necesary methods to start a new
 * widget-type. It extends from MetrodigiWidgetEditable to inheritate
 * the edditable methods for the widget.
 *
 * All custom widget classes should extend from this class.
 *
 */
MetrodigiWidget = new Class({
    Implements: [Options, Events],
    Extends: MetrodigiWidgetEditable,
    options: {
    },
    /*
     * Initializes the widget
     * @param {Array} options
     */
    initialize: function (options) {
        // Merge the default options of the Class with the options passed in
        this.setOptions(options);

        // Get widget structure elements
        this.getWidgetStructure();

        // Create the Dom
        this.createDOM();

        // Determines the clickEvent depending the device
        this.clickEvent = this.isTouchDevice() ? 'touchstart' : 'click';

        // Set touch-device attribute to true on touch devices
        if (this.isTouchDevice()) {
            $$("body").set("touch-device", "true");
        }

        // Remove the blue outline on focus
        this.removeBlueFocusOutline();

        // Call parent MetrodigiWidgetEditable
        this.parent(this.options);

        // Render the widget
        this.render();

        this.buildHotspots();
    },
    /*
     * Gets widget structure elements
     */
    getWidgetStructure: function () {
        this.container = $$(".image_container");
        this.header = this.container.getElement('.widget-header');
        this.body = this.container.getElement('.widget-body');
        this.footer = this.container.getElement('.widget-footer');
    },
    /*
     * Checks if widget is running on a touch device
     */
    isTouchDevice: function () {
        return !!('ontouchstart' in window);
    },
    /*
     * Removes the blue outline on focus
     */
    removeBlueFocusOutline: function () {
        window.onload = (function () {
            (new Element('style', {
                type: 'text/css',
                html: '*:active,*:focus{outline:none;}*:active.non-keyboard-outline,*:focus.non-keyboard-outline{outline:rgba(125,173,217,0.4) solid 2px;box-shadow:0 0 6px rgb(125,173,217);}'
            })).inject($$('head')[0]);

            window.lastKey = new Date();
            window.lastClick = new Date();
            document.addEvent('focusin', function (e) {
                $$(".non-keyboard-outline").removeClass("non-keyboard-outline");
                var wasByKeyboard = lastClick < lastKey;
                if (wasByKeyboard) {
                    e.target.addClass("non-keyboard-outline");
                }
            });
            document.addEvent('click', function () {
                window.lastClick = new Date();
            });
            document.addEvent('keydown', function () {
                window.lastKey = new Date();
            });
        });
    },
    /*
     * Builds widget's DOM
     */
    createDOM: function () {
        /*
         * This is just a function specification,
         * each widget instance has to redefine this function.
         */

    },
    /*
     * Renders the Hotspots (EDIT-78)
     */
    buildHotspots: function () {
        if (this.options.hotspots && this.options.hotspots.length > 0) {
            var _this = this;

            this.infoWrapper = new Element('div#infoWrapper');

            if ($$('.map')[0]) {
                this.infoWrapper.inject($$('.map')[0]);
            }
            else {
                this.infoWrapper.inject($$(this.container)[0]);
            }

            this.options.hotspots.each(function (popup, i) {
                position = '';

                if (typeof popup.position != 'undefined') {
                    position = popup.position;
                }

                img = new Element('button', {
                    'type': 'image',
                    'src': '../_base_pearson/images/info.png',
                    'class': 'icon-info info-' + i,
                    'data-info': 'info-' + i,
                    'aria-label': "Map callout: " + popup.text
                }).inject(_this.infoWrapper);

                popup.x = popup.x * 1;
                popup.y = popup.y * 1;

                img.setStyle('top', popup.y);
                img.setStyle('left', popup.x);

                text = new Element('span', {
                    html: popup.text,
                    'tabindex': 0,
                    'class': 'popup txt hidden info-' + i,
                    'data-info': 'info-' + i,
                    'data-position': position,
                }).inject(_this.infoWrapper);


                if (position == 'left') {
                    text.setStyle('top', popup.y + -35);
                    text.setStyle('left', popup.x - 205);

                } else if (position == 'top') {
                    text.setStyle('top', popup.y - 140);
                    text.setStyle('left', popup.x - 78);

                }
                else {
                    text.setStyle('top', popup.y + 40);
                    text.setStyle('left', popup.x - 78);

                }
            });

            $(document.body).addEvent('click', function (e) {
                if (!$(e.target).hasClass("icon-info")) {
                    $$('body span.popup').addClass('hidden');
                }
            });

            $$('body .icon-info').addEvent('click', function (e, i) {

                if ($$('body span.' + this.getAttribute('data-info'))[0].hasClass("hidden")) {
                    $$('body span.popup').addClass('hidden');
                    $$('body span.' + this.getAttribute('data-info')).removeClass('hidden');
                    //$$('body span.' + this.getAttribute('data-info'))[0].focus();
                }
                else {

                    $$('body span.popup').addClass('hidden');
                }

            });
        }
    },
    render: function () {
        /*
         * This is just a function specification,
         * each widget instance has to redefine this function.
         */
    }
});
