SymbolizeStatement = new Class({
    Implements: [Options, Events],
    initialize: function (widgetInstanceData) {

        // Set initial options here
        this.setOptions({
            title: 'Symbolize Statement Quiz',
        });

        // TEMPORARY for testing:
        // window.my = this;

        // Containers
        this.widgetContainer = document.querySelector('#widget-container');
        this.mainContainer = document.querySelector('.main-container');
        this.introContainer = document.querySelector('.intro-container');
        this.instructionContainer = document.querySelector('.instruction-container');
        this.problemContainer = document.querySelector('.problem-container');
        this.symbolContainer = document.querySelector('.symbol-container');
        this.answerContainer = document.querySelector('.answer-container');
        this.answerElement = document.querySelector('#answer-text-element');

        // Other structural elements
        this.footer = document.querySelector('footer.widget-footer');
        this.questionDisplayElement = document.querySelector('.page-status');
        this.currentQuestionElement = document.querySelector('#current-question-number');
        this.totalQuestionsElement = document.querySelector('#total-number-of-questions');
        this.footerPreviousButton = document.querySelector('footer.widget-footer button.previous');
        this.footerNextButton = document.querySelector('footer.widget-footer button.next');
        this.footerFinishButton = document.querySelector('footer.widget-footer button.finish');
        this.footerStartOverButton = document.querySelector('footer.widget-footer button.start-over');

        // Glyphs and icons (not accessible during instantiation)
        this.symbolGlyphs = [];
        this.inputGlyphs;
        this.answerGlyphs;

        // State and score data
        this.widgetInstanceData = widgetInstanceData;
        this.currentQuestionNumber = 1;
        this.totalQuestions = widgetInstanceData.questionData.length;
        this.submittedAnswers = {};
        this.totalScore = 0;
        this.questionAttempts = [0];
        this.immediateFeedbackHasBeenDismissed = true;

        // Remaining initialization operations
        this.updateFooterButtonText();
        this.selectLogicSymbols();
        this.updateFooterElements();
        this.updateTextContent(true);
        this.bindEvents();
        this.removeSpacesFromAnswers();
    }, // end initialize()


    /**
     * If this widget is in 'immediate feedback" mode,
     * change the button text from "Next" to "Check answer".
     **/
    updateFooterButtonText: function () {
        if (this.widgetInstanceData.requireImmediateFeedback === true) {
            this.footerNextButton.textContent = 'Check answer';
        }
    }, // end updateFooterButtonText()


    /**
     * Bind all DOM events
     **/
    bindEvents: function () {
        this.footerPreviousButton.addEventListener('click', this.previousButtonHandler.bind(this));
        this.footerNextButton.addEventListener('click', this.nextButtonHandler.bind(this));
        this.footerFinishButton.addEventListener('click', this.finishButtonHandler.bind(this));
        this.footerStartOverButton.addEventListener('click', this.startOverButtonHandler.bind(this));
        this.answerElement.addEventListener('keydown', this.answerElementKeyboardInputHandler.bind(this));
    }, // end bindEvents()


    /**
     * Populate or update text content based on current question.
     *   @BOOL: populateIntroText - if "true" will populate the intro text box
     * TODO: If we want a "start" page, set current page to "0" and load DefaultText
     **/
    updateTextContent: function (populateIntroText) {
        this.crossfadeText(this.problemContainer, (this.widgetInstanceData.questionData[this.currentQuestionNumber - 1].question));

        if (!!(this.submittedAnswers[this.currentQuestionNumber])) {
            this.crossfadeText(this.answerElement, (this.submittedAnswers[this.currentQuestionNumber]));
        } else {
            this.crossfadeText(this.answerElement, '');
        }

        if (!!populateIntroText) {
            this.introContainer.textContent = this.widgetInstanceData.introContainerText;
            this.instructionContainer.textContent = this.widgetInstanceData.instructionContainerText;
        }

    }, // end updateTextContent()

    /**
     * Just a nice little crossfade text effect
     **/
    crossfadeText: function (element, newText) {
        if (element.tagName.toUpperCase() === 'INPUT') {
            element.addClass('text-fade');
            window.setTimeout(function () {
                element.value = newText;
                element.removeClass('text-fade');
            }, 120);
        } else {
            element.addClass('text-fade');
            window.setTimeout(function () {
                element.innerHTML = newText;
                element.removeClass('text-fade');
            }, 90);
        }
    }, // end crossfadeText()


    /**
     * Populate symbol glyphs by:
     *   First: Rendering all symbols in the "extraLogicSymbolsToRender" variable
     *   Second: choosing which logical symbols from "logicSymbols" get rendered to the screen
     *   Third: creating the symbol glyph element.
     *
     * If this.widgetInstanceData.autoSelectLogicSymbols is "true", this will only
     * use symbols that appear in the answers, otherwise will use all available.
     **/
    selectLogicSymbols: function () {
        var extraLogicSymbolsString = this.widgetInstanceData.extraLogicSymbolsToRender.toString();

        /**
         * Create all symbols in "extraLogicSymbolsToRender"
         **/
        if (!!this.widgetInstanceData.extraLogicSymbolsToRender) {
            for (var i = 0; i < this.widgetInstanceData.extraLogicSymbolsToRender.length; i++) {
                this.populateSymbolGlyph(this.widgetInstanceData.extraLogicSymbolsToRender[i]);
            } // end for (widgetInstanceData.extraLogicSymbolsToRender)
        } // end (extraLogicSymbolsToRender is not empty)

        /**
         * Iterate through all the available symbols
         **/
        for (var i = 0; i < this.widgetInstanceData.logicSymbols.length; i++) {
            var searchTermRegExp = '';

            // Catch RegExp errors and fix by prepending with '\'.
            try {
                searchTermRegExp = new RegExp((this.widgetInstanceData.logicSymbols[i]));
            } catch (error) {
                // console.warn(this.widgetInstanceData.logicSymbols[i], error);
                fixedSymbolChar = '\\' + this.widgetInstanceData.logicSymbols[i];
                searchTermRegExp = new RegExp(fixedSymbolChar);
            }

            // If this symbol already appears in extraLogicSymbolsToRender, continue this loop
            var extraSymbolsContainsThisSymbol = extraLogicSymbolsString.search(searchTermRegExp);

            if (extraSymbolsContainsThisSymbol !== -1) {
                console.warn('The symbol "' + (this.widgetInstanceData.logicSymbols[i]) + '" was already rendered in extraLogicSymbols. Continuing this loop...');
                continue;
            }

            /**
             * If not auto-selecting, immediately populate symbol
             **/
            if (!this.widgetInstanceData.autoSelectLogicSymbols) {
                this.populateSymbolGlyph(this.widgetInstanceData.logicSymbols[i]);
            } else {
                /**
                 * We are auto-selecting, so iterate through all the provided answers
                 **/
                for (var j = 0; j < this.widgetInstanceData.questionData.length; j++) {

                    // Search answer for symbol usage
                    var answerContainsThisSymbol = (this.widgetInstanceData.questionData[j].answer).search(searchTermRegExp);

                    // If this symbol is found in an answer add the glyph and break this loop
                    if (answerContainsThisSymbol !== -1) {
                        this.populateSymbolGlyph(this.widgetInstanceData.logicSymbols[i]);
                        break;
                    }
                } // end for (all answers)
            } // end else (auto-select symbol glyphs)
        } // end for (iterate through all logicSymbols)
    }, // end selectLogicSymbols()


    /**
     * @STRING symbol - The symbol to render to the symbol container
     */
    populateSymbolGlyph: function (symbol) {
        var glyphElement = new Element('div');
        var glyphID = 'glyph-' + symbol;

        glyphElement.addClass('glyph');
        glyphElement.addClass('glyph-input');
        glyphElement.addClass('text-unselectable');
        glyphElement.setAttribute('id', glyphID);
        glyphElement.textContent = symbol;
        glyphElement.addEventListener('click', this.inputGlyphHandler.bind(this));

        this.symbolGlyphs.push(glyphElement);
        this.symbolContainer.appendChild(this.symbolGlyphs[(this.symbolGlyphs.length - 1)]);
    }, // end populateSymbolGlyphs()


    /**
     * Populate, hide, or update page numbers and footer elements
     *   @BOOL hideElementFlag = flag to hide this display and "Prev" button
     **/
    updateFooterElements: function (hideElementFlag) {
        if (hideElementFlag) {
            // This is the results page: hide "Previous" and page guides
            this.questionDisplayElement.hide();
            this.footerPreviousButton.hide();
        } else {
            // Any page besides the results page.
            this.currentQuestionElement.textContent = this.currentQuestionNumber;
            this.totalQuestionsElement.textContent = this.totalQuestions;

            // First question
            if (this.currentQuestionNumber === 1) {
                this.footerPreviousButton.hide();
                this.footerFinishButton.hide();
                this.footerNextButton.show();
            } else if (this.currentQuestionNumber === this.totalQuestions &&
                this.widgetInstanceData.requireImmediateFeedback !== true) {
                // If this is the final question without requireImmediateFeedback
                this.footerNextButton.hide();
                this.footerFinishButton.show();
                this.footerPreviousButton.show();
            } else if (this.widgetInstanceData.requireImmediateFeedback === true) {
                // If this is any other question WITH requireImmediateFeedback
                this.footerFinishButton.hide();
                this.footerPreviousButton.hide();
                this.footerNextButton.show();
            } else {
                // If this is any other question WITHOUT requireImmediateFeedback
                this.footerFinishButton.hide();
                this.footerPreviousButton.show();
                this.footerNextButton.show();
            }
        }
    }, // end updateFooterElements()

    /*****************************************************
     * Events for previous, next, finish, input, glyphs. *
     *****************************************************/
    /**
     * Depending on the value of widgetInstanceData.requireImmediateFeedback,
     * the Next button will either assess the score immediately or
     * move to the next question.
     */
    nextButtonHandler: function (event) {
        // Instantiate current element in questionAttempts
        if (this.questionAttempts[this.currentQuestionNumber] === undefined) {
            this.questionAttempts[this.currentQuestionNumber] = 0;
        }

        // If this instance requires immediate feedback:
        if (this.widgetInstanceData.requireImmediateFeedback === true &&
            this.questionAttempts[this.currentQuestionNumber] < (this.widgetInstanceData.maxAttempts + 1)) {

            // Remove any feedback elements on screen (if user clicked before the previous element was removed)
            //            this.removeFeebackElements();

            // Increment currentQuestionNumber
            this.questionAttempts[this.currentQuestionNumber]++;

            // If there are any guesses left
            if (this.questionAttempts[this.currentQuestionNumber] <= this.widgetInstanceData.maxAttempts) {
                // Assess the answer
                var submittedAnswer = this.answerElement.value;
                var correctAnswer = this.widgetInstanceData.questionData[this.currentQuestionNumber - 1].answer;
                this.immediateFeedbackHasBeenDismissed = false;

                // Modify strings if "answersAreCaseSensitive" flag is "false"
                if (this.widgetInstanceData.answersAreCaseSensitive === false) {
                    submittedAnswer = submittedAnswer.toUpperCase();
                    correctAnswer = correctAnswer.toUpperCase();
                } // end if (case insensitive)
                
                // Handle correct answer
                
                
                if (submittedAnswer === correctAnswer) {
                    this.answerElement.setAttribute('data-assessment-value', 'correct');
                    this.answerElement.addClass('correct');
                    this.answerElement.value += ' ✓';

                    window.setTimeout(this.removeFeebackElements.bind(this), this.widgetInstanceData.feedbackTextDuration);
                } else {
                    // Incorrect answer
                    this.answerElement.setAttribute('data-assessment-value', 'incorrect');
                    this.answerElement.addClass('incorrect');
                    this.answerElement.value += ' ✗';

                    window.setTimeout(this.removeFeebackElements.bind(this), this.widgetInstanceData.feedbackTextDuration);
                } // end (incorrect answer)

            } // end (not after last guess in immediateFeedback)
        } else {
            // No immediate feedback is required
            this.goToNextPage();
        } // end if (widget doesn't require immediate feedback)
    }, // end nextButtonHandler()

    /**
     * If feedback has not yet been dismissed, dismiss it and handle buttons.
     */
    removeFeebackElements: function (isFromKeyboardEvent) {
        if (!this.immediateFeedbackHasBeenDismissed) {
            var classToRemove = this.answerElement.getAttribute('data-assessment-value');
            this.answerElement.setAttribute('data-assessment-value', '');

            // console.log('isFromKeyboardEvent =', isFromKeyboardEvent, '    classToRemove =', classToRemove);

            this.immediateFeedbackHasBeenDismissed = true;
            this.answerElement.value = this.answerElement.value.slice(0, -2);
            this.answerElement.removeClass(classToRemove);

            // Handle different logic based on 'correct' or 'incorrect'
            if (classToRemove === 'correct') {

                // If this is the last question, call finishButtonHandler()
                if (this.currentQuestion === this.totalQuestions) {} else {
                    this.goToNextPage();
                } // end (not the last question)    
                //            } else if (classToRemove === 'incorrect' && !isFromKeyboardEvent) {
            } else if (classToRemove === 'incorrect') {
                // if this is the last guess, set flag and move on
                if (this.questionAttempts[this.currentQuestionNumber] === this.widgetInstanceData.maxAttempts) {
                    this.goToNextPage();
                } // end if (last attempt)
                //            } else if (classToRemove === 'incorrect' && !!isFromKeyboardEvent) {
            } else if (classToRemove === 'incorrect') {
                // console.log('keyboard event detected ');
            } else {
                console.error('ERROR: variable classToRemove has invalid value:', classToRemove);
            }
        } // end if (!this.immediateFeedbackHasBeenDismissed)
    }, // end removeFeebackElements(classToRemove)


    /**
     * Previous button, which is NOT available in
     * 'immediate feedback' mode.
     */
    previousButtonHandler: function (event) {
        // Save answer
        this.submittedAnswers[this.currentQuestionNumber] = this.answerElement.value;

        // If we are not <= 0, update the display
        if (--this.currentQuestionNumber) {
            this.updateTextContent();
            this.updateFooterElements();
        } else {
            // if this.currentQuestionNumber === 0, reset to 1
            this.currentQuestionNumber = 1;
            //            this.updateTextContent();
        }
    }, // end previousButtonHandler()

    /**
     * Finish button, which is NOT available in
     * 'immediate feedback' mode.
     */
    finishButtonHandler: function (event) {
        // Save answer
        this.submittedAnswers[this.currentQuestionNumber] = this.answerElement.value;

        // If we are not <= 0, update the display
        this.updateFooterElements(true);
        this.calculateScore();
    }, // end finishButtonHandler()


    /**
     * This is here to catch keystrokes and/or prevent spaces.
     **/
    answerElementKeyboardInputHandler: function (event) {

        // On keyboard events, remove feedback elements.
        if (!this.immediateFeedbackHasBeenDismissed) {
            this.removeFeebackElements(true);
        }

        if (event.defaultPrevented) {
            return; // Should do nothing if the key event was already consumed.
        } else if (event.keyCode === 13) { // enter key
            // If the user hits "enter" click "next" or "finish"
            var forwardButton = document.querySelector('footer > button');
            forwardButton.click();
        } else if (event.keyCode === 32) { // space key
            event.target.removeClass('non-keyboard-outline');
            event.preventDefault();
            event.target.addClass('bad-input');
            window.setTimeout(function () {
                event.target.removeClass('bad-input');
            }, 150, event);
        }
    }, // end answerElementKeyboardInputHandler()


    /**
     * Save answer, update DOM and render next page.
     **/
    goToNextPage: function () {
        // Save answer
        this.submittedAnswers[this.currentQuestionNumber] = this.answerElement.value;

        // If we are not <= 0, update the display
        if (this.currentQuestionNumber++ <= this.totalQuestions) {

            // After the last question is answered:
            if (this.currentQuestionNumber === (this.totalQuestions + 1)) {
                this.finishButtonHandler();
                this.updateFooterElements(true);
            } else {
                this.updateTextContent();
                this.updateFooterElements();
            } // not the last question
        } else {
            this.currentQuestionNumber--;
        }
    }, // end goToNextPage()

    /**
     * Change input field value when text
     * is selected and caret is not at end.
     **/
    inputGlyphHandler: function (event) {
        // Get caret / selection positions
        var caretStartPosition = this.answerElement.selectionStart;
        var caretEndPosition = this.answerElement.selectionEnd;

        // Store the text before and after the carat / selection
        var textBeforeCaret = this.answerElement.value.slice(0, caretStartPosition);
        var textAfterCaret = this.answerElement.value.slice(caretEndPosition, (this.answerElement.value.length));

        // Update the text value with the glyph in the right location
        this.answerElement.value = textBeforeCaret + event.target.textContent + textAfterCaret;

        // Set the carat to the end of the glyph position.
        caretStartPosition++;
        this.answerElement.selectionStart = caretStartPosition;
        this.answerElement.selectionEnd = caretStartPosition;
    }, // end inputGlyphHandler()

    /**
     * Update the DOM after the "Finish" button is pressed.
     * Then calculate the score and add feedback elements.
     * 
     * Submitted answers are in this.submittedAnswers[questionNumber]
     *
     * Once the widget is sccored the score can be accessed by:
     *   @INT this.totalScore = number of correct answers
     *   @INT this.totalQuestions = number of possible correct answers
     **/
    calculateScore: function () {
        /**
         * Update the DOM text and elements:
         **/
        this.answerElement.hide();
        this.symbolContainer.hide();
        this.footerFinishButton.hide();
        this.footerNextButton.hide();
        this.footerStartOverButton.show();

        if (this.widgetInstanceData.hideIntroTextOnFinalPage) {
            this.introContainer.hide();
            this.instructionContainer.hide();
        }

        this.problemContainer.addClass('score-text');
        this.problemContainer.textContent = 'Your final score is:';

        // var answerFeedbackContainer = new Element('table');
        var answerFeedbackContainer = new Element('div');
        answerFeedbackContainer.setAttribute('id', 'answer-feedback-container');
        answerFeedbackContainer.addClass('feedback-container');
        //        answerFeedbackContainer.addClass('flex-container');

        var scoreDisplayElement = new Element('h2');
        scoreDisplayElement.setAttribute('id', 'score-display-element');

        /**
         * Iterate over submitted answers and compare to widgetInstanceData answers.
         * Construction of assessment elements is:
         * 
         *     #answer-feedback-container
         *         .assessment-row
         *             .assessment-question-label
         *             
         *             .assessment-feedback-container
         *                 .answer-glyph
         *                 .your-response-text
         *                 .submitted-answer-element
         *                 .correct-response-text
         *                 .correct-answer-element
         **/
        for (var i = 0; i < this.widgetInstanceData.questionData.length; i++) {
            var thisQuestion = this.widgetInstanceData.questionData[i].question;
            var submittedAnswer = this.submittedAnswers[i + 1];
            var correctAnswer = this.widgetInstanceData.questionData[i].answer;

            var assessmentRow = new Element('div');
            assessmentRow.addClass('assessment-row');

            var assessmentQuestionLabel = new Element('div');
            assessmentQuestionLabel.addClass('assessment-question-label');
            assessmentQuestionLabel.innerHTML = (i + 1) + '. ' + thisQuestion;

            var assessmentFeedbackContainer = new Element('div');
            assessmentFeedbackContainer.addClass('assessment-feedback-container');

            var thisAnswerGlyph = new Element('div');
            thisAnswerGlyph.addClass('answer-glyph');
            thisAnswerGlyph.textContent = '✓';

            var yourResponseTextElement = new Element('div');
            yourResponseTextElement.addClass('your-response-text');
            yourResponseTextElement.textContent = 'Your response: ';

            var submittedAnswerElement = new Element('div');
            submittedAnswerElement.addClass('submitted-answer-element');
            submittedAnswerElement.textContent = submittedAnswer;

            var correctResponseTextElement = new Element('div');
            correctResponseTextElement.addClass('correct-response-text');
            correctResponseTextElement.textContent = 'Correct response: ';

            var correctAnswerElement = new Element('div');
            correctAnswerElement.addClass('correct-answer-element');
            correctAnswerElement.textContent = correctAnswer;

            if (correctAnswer === submittedAnswer) {
                this.answerIsCorrect(submittedAnswerElement, thisAnswerGlyph, correctAnswerElement, correctResponseTextElement);
            } else if ((correctAnswer.toUpperCase()) === (submittedAnswer.toUpperCase())) {
                // Check for case-insensitive equality.
                console.warn('WARNING: This answer may be correct, but the upper-case and lower-case is incorrect.');
                
                if (this.widgetInstanceData.answersAreCaseSensitive === false) {
                    this.answerIsCorrect(submittedAnswerElement, thisAnswerGlyph, correctAnswerElement, correctResponseTextElement);
                } else {
                    this.answerIsIncorrect(submittedAnswerElement, thisAnswerGlyph, correctAnswerElement);                  
                }
            } else {
                this.answerIsIncorrect(submittedAnswerElement, thisAnswerGlyph, correctAnswerElement);
            }

            answerFeedbackContainer.appendChild(assessmentRow);

            assessmentRow.appendChild(assessmentQuestionLabel);
            assessmentRow.appendChild(assessmentFeedbackContainer);

            assessmentFeedbackContainer.appendChild(thisAnswerGlyph);
            assessmentFeedbackContainer.appendChild(yourResponseTextElement);
            assessmentFeedbackContainer.appendChild(submittedAnswerElement);
            assessmentFeedbackContainer.appendChild(correctResponseTextElement);
            assessmentFeedbackContainer.appendChild(correctAnswerElement);
        } // end for (all questions)

        scoreDisplayElement.textContent = '' + this.totalScore + ' / ' + this.totalQuestions;
        this.problemContainer.appendChild(scoreDisplayElement);
        this.problemContainer.appendChild(answerFeedbackContainer);
    }, // end calculateScore()


    /**
     * Handle correct answers
     **/
    answerIsCorrect: function (submittedAnswerElement, thisAnswerGlyph, correctAnswerElement, correctResponseTextElement) {
        this.totalScore++;
        submittedAnswerElement.addClass('correct');
        thisAnswerGlyph.addClass('correct');
        correctAnswerElement.addClass('hidden');
        correctResponseTextElement.addClass('hidden');
    },

    
    /**
     * Handle incorrect answers
     **/
    answerIsIncorrect: function (submittedAnswerElement, thisAnswerGlyph, correctAnswerElement) {
        submittedAnswerElement.addClass('incorrect');
        thisAnswerGlyph.addClass('incorrect');
        thisAnswerGlyph.textContent = '✗';
        correctAnswerElement.addClass('correct');  
    },


    /**
     * Restart this widget by refreshing the page
     **/
    startOverButtonHandler: function () {
        document.location.reload();
    },


    /**
     * Normalize answers by removing all spaces.
     **/
    removeSpacesFromAnswers: function () {
        for (var i = 1; i <= this.widgetInstanceData.questionData.length; i++) {
            var correctAnswerWithoutSpaces = (this.widgetInstanceData.questionData[i - 1].answer).replace(' ', '');

            while ((this.widgetInstanceData.questionData[i - 1].answer) !== correctAnswerWithoutSpaces) {
                // Check for case-insensitive equality.
                this.widgetInstanceData.questionData[i - 1].answer = correctAnswerWithoutSpaces;
                correctAnswerWithoutSpaces = (this.widgetInstanceData.questionData[i - 1].answer).replace(' ', '');
                console.warn('WARNING: This answer had spaces removed:', correctAnswerWithoutSpaces);
            }
        } // end for (all questions)
    },


    /**
     * Render all content on page.
     * Currently unused.
     */
    render: function () {} // end render()
}); // end SymbolizeStatement()