/**
 * Step-by-step walkthrough of a sequence with a top location indicator.
 *
 * Author: Aaron Melocik, (adapted from Unknown)
 * Date of major revision: 4.6.2015
 * Sample widget reference: ...
 *
 *
 * WalkthroughSlider constructor takes:                                                     OPTIONAL PARAMETERS:
 * {instructions, startingSlide, widgetContent[{}, {}, etc.] [[, canvasWidth, canvasHeight]]}
 */
WalkthroughSlider = new Class({
    Implements: [Options, Events],
    Extends: MetrodigiWidget,

    // Set initial options here
    options: {
        instructions: '<p>Click each point on the sequence line.</p>',
        startingSlide: 0
    },
    slide: 0,

    // initialize - widget initialization
    initialize: function (options) {
        var _this = this;

        this.headerElement = document.getElementsByClassName('widget-header')[0];
        this.widgetBodyElement = document.getElementsByClassName('widget-body')[0];

        this.contentAreaElement;
        this.sliderContainerElement;
        this.navListElement;
        this.navElements = [];
        this.slideElements = [];
        this.currentSlide;

        // this.parent(options) invokes createDOM() as a part of MetrodigiWidget parent class instantiation
        this.parent(options);

        this.container.getElements('.nav-li').addEvent(this.clickEvent, function (event) {
            for (var i = 0; i < _this.navElements.length; i++) {
                _this.navElements[i].removeClass('clicked');
            }
            
            this.addClass('clicked');
            _this.slide = this.dataset.slide_selector.toInt();
            _this.render();
        });
    }, // end initialize()

    createInstructionHeader: function () {
        var _this = this;

        _this.instructionsElement = new Element('div.instructions', {});

        _this.instructionsElement.innerHTML = _this.options.instructions;
        _this.instructionsElement.inject(_this.headerElement);
    }, // end createInstructionHeader()

    /**
     * Create the DOM, organized as follows:
     *
     * <body>
     *     <div class="container">
     *         <div class="MetrodigiWidget">
     *             <main class="widget-content">
     *                 <header class="widget-header">
     *                 |   <div class="instructions">
     *                 <article class="widget-body">
     *                 |   <nav class="slider-container">
     *                 |       <ul class="nav-list flex-container">
     *                 |       |   <li class="nav-li clickable [[clicked]]">
     *                 |       |       <div class="mask">
     *                 |   <div class="content-area">
     *                 |       <section class="slide">
     *                 |       |   <div class="text-content">
     *                 |       |       <h5 class="slide-header text-header">
     *                 |       |   <figure class="image-container">
     *                 |       |       <h5 class="slide-header image-header">
     *                 |       |       <img class="image-content">
     *                 <footer class="widget-footer"></footer>
     **/
    createDOM: function () {
        var _this = this;
        var slideUnderConstruction;
        _this.slide = _this.options.startingSlide;

        this.createInstructionHeader();

        _this.sliderContainerElement = new Element('nav.slider-container').inject(_this.widgetBodyElement);
        _this.contentAreaElement = new Element('div.content-area').inject(this.body);
        _this.navListElement = new Element('ul.nav-list.flex-container').inject(_this.sliderContainerElement);

        // iterate over JSON data to build DOM elements
        for (var i = 0; i < this.options.data.length; i++) {
            slideUnderConstruction = this.options.data[i];

            // top-level section element for each block:
            var sectionSlide = new Element('section', {
                'class': 'slide transparent',
                'data-slide': i
            });

            // all text content for this slide:
            var textContent = new Element('div.content-container.text-content', {
                'aria-live': 'rude'
            }).inject(sectionSlide);

            new Element('h5.slide-header.text-header', {
                html: slideUnderConstruction.sectionTitle,
                'role': 'heading'
            }).inject(textContent);

            new Element('p', {
                html: slideUnderConstruction.sectionContent
            }).inject(textContent);

            // the image content for this slide:
            var imageContainer = new Element('figure.content-container.image-container').inject(sectionSlide);
            var imagePath;

            // if an image path is supplied, use it. Otherwise, try the default "slide[[i+1]].png".
            if (!!slideUnderConstruction.sectionFigure && slideUnderConstruction.sectionFigure !== " ") {
                imagePath = 'images/' + slideUnderConstruction.sectionFigure;
            } else {
                try {
                    imagePath = 'images/slide' + (i + 1) + '.png';
                } catch (e) {
                    console.log('Error in assigning imagePath', (i + 1), ':', e);
                }
            } // end if-else

            new Element('h5.slide-header.image-header', {
                html: slideUnderConstruction.figureTitle,
                'role': 'heading'
            }).inject(imageContainer);
            
            new Element('img.image-content', {
                'src': imagePath
            }).inject(imageContainer);

//            if (!!slideUnderConstruction.figureTitle && slideUnderConstruction.figureTitle !== " ") {
//                new Element('h3.image-title', {
//                    html: slideUnderConstruction.figureTitle,
//                    'role': 'heading'
//                }).inject(this.imageContainer);
//            }

            if (!!slideUnderConstruction.figureSubtitle) {
                new Element('h4.image-subtitle', {
                    html: slideUnderConstruction.figureSubtitle,
                    'role': 'heading'
                }).inject(imageContainer);
            };

            if (slideUnderConstruction.freeText) {
                for (var j = 0; j < slideUnderConstruction.freeText.length; j++) {
                    new Element('span.free-text', {
                        html: slideUnderConstruction.freeText[j],
                        'data-id': j
                    }).inject(imageContainer);
                };
            }

            // navigation li elements
            var navListItemElement = new Element('li', {
                class: 'nav-li clickable',
                id: 'nav-li-' + i,
                'data-slide_selector': i,
                tabindex: 0,
                events: {
                    keyup: function (e) {
                        if (e.key == "tab") {
                            this.fireEvent('click');
                            _this.render();
                        }
                    }
                }
            });

            var mask = new Element('div.mask').inject(navListItemElement);

            if (!!slideUnderConstruction.navigationLineTitle) {
                    var navText = new Element('div.nav-text-tooltip', {
                    'text': slideUnderConstruction.navigationLineTitle,
                }).inject(mask);
    //            }).inject(navListItemElement); //???
    //            Possibly inject this elsewhere to fix hovering tooltip issue.
            }

            _this.navElements.push(navListItemElement);
            _this.slideElements.push(sectionSlide);

            // if this is the first slide register the nav-li el as 'clicked' and make the slide visible.
            if (i === _this.options.startingSlide) {
                navListItemElement.addClass('clicked');
                _this.slideElements[i].addClass('opaque');
                _this.slideElements[i].removeClass('transparent');
                _this.currentSlide = _this.slideElements[i];
            }

            _this.navElements[i].inject(_this.navListElement);
            _this.slideElements[i].inject(_this.contentAreaElement);
        }; // end for (iterate through DOM elements creation loop)
    }, // end createDOM()

    // render - updates the widget display
    render: function () {
        var _this = this;

        var elementsToMakeTransparent = this.container.getElements('[data-slide]');
        _this.currentSlide = _this.slideElements[_this.slide];

        elementsToMakeTransparent.addClass('transparent');
        _this.currentSlide.addClass('opaque');

        try {
            elementsToMakeTransparent.removeClass('opaque');
            _this.currentSlide.removeClass('transparent');
        } catch (e) {
            console.log('Error in opacitiy transition for slide', _this.slide, ':\n', e);
        }
    } // end render()
});
